#region << Using Directives >>
using System;
using System.IO;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;
using Volpe.XlLib;
#endregion

namespace Volpe.Cafe.IO.Reporting.XL
{
    /// <summary>
    /// Provides a base class for generating Excel reports.
    /// </summary>
    [Serializable]
    public abstract class XlReportingBase
	{

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="XlReportingBase"/> class, using the specified parameters.
        /// </summary>
        /// <param name="path">The output path for the report.</param>
        /// <param name="outputSettings">The output settings to use for encrypting the report.</param>
        /// <param name="templateFile">An Excel file to use as a template for the report.</param>
        /// <param name="templateName">The name of the template worksheet, within the tempalteFile, to use as a template for the
        ///   report.</param>
        public XlReportingBase(string path, OutputSettings outputSettings, XlUtilities templateFile, string templateName)
        {
            this.InitReport(path, outputSettings);
            this.InitTemplate(templateFile, templateName);
            //
            this._wsOpen   = false;
            this._wsCount  = 0;
            this._disposed = false;
        }

        #endregion


        #region /*** Methods ***/

        void InitReport(string path, OutputSettings outputSettings)
        {
            if (File.Exists(path)) { File.Delete(path); }
            this._xlu = new XlUtilities(path, false, null, null, XlCryptoSettings.Empty);
            this._path = path;
        }
        void InitTemplate(XlUtilities templateFile, string templateName)
        {
            this._templateFile = templateFile;
            this._templateName = templateName;
            object[,] tBuf = templateFile.GetData(templateName, true);
            //
            this._tRows = tBuf.GetUpperBound(0);
            this._tCols = tBuf.GetUpperBound(1);
            this._template = new object[this._tRows, this._tCols];
            for (int i = 0; i < this._tRows; i++)
            {
                for (int j = 0; j < this._tCols; j++)
                {   // excel uses 1-based indexing when returning data, hence tBuf is offset by 1
                    this._template[i, j] = tBuf[i + 1, j + 1];
                }
            }
        }

        /// <summary>
        /// Parses the data into the report.
        /// </summary>
        /// <param name="scen">The scenario to report on.</param>
        /// <param name="year">The model year to report on.</param>
        /// <param name="scenData">The modeling data, as it was at the end of the scenario and the model year being reported.</param>
        /// <param name="baseData">The modeling data, as it was at the end of the baseline scenario and the model year being reported.</param>
        /// <param name="ppData">The product plan modeling data, as it was loaded from the input file.</param>
        public void ParseData(Scenario scen, ModelYear year, Industry scenData, Industry baseData, Industry ppData)
        {
            try
            {
                this.ParseData_Internal(scen, year, scenData, baseData, ppData);
            }
            catch (Exception ex)
            {
                if (this._wsOpen) { this.CloseBuffer(); }
                this.Close(false);
                //
                throw new OutputException(this.FileName, null, ex);
            }
        }
        /// <summary>
        /// Parses the data into the report.
        /// </summary>
        /// <param name="scen">The scenario to report on.</param>
        /// <param name="year">The model year to report on.</param>
        /// <param name="scenData">The modeling data, as it was at the end of the scenario and the model year being reported.</param>
        /// <param name="baseData">The modeling data, as it was at the end of the baseline scenario and the model year being reported.</param>
        /// <param name="ppData">The product plan modeling data, as it was loaded from the input file.</param>
        protected abstract void ParseData_Internal(Scenario scen, ModelYear year, Industry scenData, Industry baseData, Industry ppData);

        /// <summary>
        /// Opens a new buffer for writing.
        /// </summary>
        /// <param name="bufName">The name to assign to the buffer.</param>
        public void OpenBuffer(string bufName)
        {
            this.OpenBuffer(bufName, true);
        }
        /// <summary>
        /// Opens a new buffer for writing.
        /// </summary>
        /// <param name="bufName">The name to assign to the buffer.</param>
        /// <param name="copyTemplateData">true, to copy header data from the template sheet; false, to initialize an empty buffer.</param>
        public void OpenBuffer(string bufName, bool copyTemplateData)
        {
            this.OpenBufferInternal(bufName, false, XlSize.Empty, copyTemplateData);
        }
        /// <summary>
        /// Opens a new buffer for writing, using the specified buffer size.
        /// </summary>
        /// <param name="bufName">The name to assign to the buffer.</param>
        /// <param name="size">The size of the internal 2D array used as the worksheet buffer.</param>
        public void OpenBuffer(string bufName, XlSize size)
        {
            this.OpenBuffer(bufName, size, true);
        }
        /// <summary>
        /// Opens a new buffer for writing, using the specified buffer size.
        /// </summary>
        /// <param name="bufName">The name to assign to the buffer.</param>
        /// <param name="size">The size of the internal 2D array used as the worksheet buffer.</param>
        /// <param name="copyTemplateData">true, to copy header data from the template sheet; false, to initialize an empty buffer.</param>
        public void OpenBuffer(string bufName, XlSize size, bool copyTemplateData)
        {
            this.OpenBufferInternal(bufName, true, size, copyTemplateData);
        }
        void OpenBufferInternal(string bufName, bool useCustomSize, XlSize customSize, bool copyTemplateData)
        {
            if (this._disposed) { throw new XlReportingException("The report was already closed.", this._path); }
            if (this._wsOpen) { throw new XlReportingException("A buffer was already opened for output.", this._path); }
            //
            this._wsOpen = true;
            //
            // initialize buffer & copy data from template
            this._bRows = (useCustomSize) ? Math.Max(customSize.Rows   , this._tRows) : this._tRows;
            this._bCols = (useCustomSize) ? Math.Max(customSize.Columns, this._tCols) : this._tCols;
            this._buffer = new object[this._bRows, this._bCols];
            //
            if (copyTemplateData)
            {
                if (useCustomSize)
                {
                    for (int i = 0; i < this._tRows; i++)
                    {
                        for (int j = 0; j < this._tCols; j++)
                        {
                            this._buffer[i, j] = this._template[i, j];
                        }
                    }
                }
                else
                {
                    Array.Copy(this._template, this._buffer, this._template.Length);
                }
            }
            // copy template worksheet
            if (this._wsCount == 0) { this._xlu.DeleteAllWorksheets(); }    // removes all "default" Excel sheets
            this._xlu.CopyWorksheet(this._templateFile, this._templateName, ++this._wsCount);
            this._xlu.SetWorksheetName(this._wsCount, bufName);
            if (this._wsCount == 1) { this._xlu.DeleteWorksheet(2); }       // removes the last "default" Excel sheet
        }
        /// <summary>
        /// Closes the active buffer, writing any changes to the workbook.
        /// </summary>
        public void CloseBuffer()
        {
            this.CloseBuffer(true, new XlRange(1, 1, this._bRows, this._bCols));
        }
        /// <summary>
        /// Closes the active buffer, writing any changes to the workbook.
        /// </summary>
        /// <param name="autoFitText">Specifies whether to auto-fit the text within the buffer.</param>
        /// <param name="autoFitRange">Specifies the range of cells within the buffer to auto-fit.</param>
        public void CloseBuffer(bool autoFitText, XlRange autoFitRange)
        {
            if (this._disposed) { throw new XlReportingException("The report was already closed.", this._path); }
            if (!this._wsOpen) { throw new XlReportingException("A buffer has not been opened for output.", this._path); }
            //
            this._xlu.SetData(this._wsCount, this._buffer);
            if (autoFitText)
            {
                this._xlu.AutoFitText(autoFitRange.Cell1, autoFitRange.Cell2);
            }
            //
            this._wsOpen = false;
        }
        /// <summary>
        /// Closes the report, optionally saving any changes.
        /// </summary>
        /// <param name="saveChanges">Specifies whether to save the changes made to the report before closing.</param>
        public void Close(bool saveChanges)
        {
            if (this._disposed) { throw new XlReportingException("The report was already closed.", this._path); }
            if (this._wsOpen) { throw new XlReportingException("All output buffers must be closed before saving the report.", this._path); }
            //
            this._xlu.ActivateWorksheet(1);
            this._xlu.Close(saveChanges);
            //
            this._disposed = true;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets whether the report has been closed and its resources released.</summary>
        public bool Disposed { get { return this._disposed; } }

        /// <summary>Gets the location of the report.</summary>
        public string Path { get { return this._path; } }
        /// <summary>Gets the filename of the report.</summary>
        public string FileName { get { return System.IO.Path.GetFileName(this._path); } }

        /// <summary>Gets the number of rows in the buffer.</summary>
        protected int Rows { get { return this._bRows; } }
        /// <summary>Gets the number of columns in the buffer.</summary>
        protected int Cols { get { return this._bCols; } }

        /// <summary>Gets whether a worksheet buffer is open for writing.</summary>
        protected bool WSOpen { get { return this._wsOpen; } }

        #endregion


        #region /*** Variables ***/

        XlUtilities _templateFile;
        string      _templateName;
        object[,]   _template;
        int         _bRows, _bCols, _tRows, _tCols;     // rows/cols in the buffer and template

        string _path;
        XlUtilities _xlu;
        /// <summary>The active buffer, which contains data to be written to the report.
        ///   Call <see cref="OpenBuffer(string, bool)"/> to initialize the buffer,
        ///   then <see cref="CloseBuffer()"/> to save the changes.</summary>
        protected object[,] _buffer;

        bool _wsOpen;
        int  _wsCount;

        bool _disposed;     // set to true after the report has been closed

        #endregion

	}
}
