#region << Using Directives >>
using System;
using System.Collections;
using System.Collections.Generic;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.Settings;
using TI = Volpe.Cafe.TechnologyIndexes;
using TA = Volpe.Cafe.Model.TechnologyApplicability;
using Volpe.Cafe.Utils;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides a linear, brute-force approach for generating compliance finding solutions.
    /// </summary>
    [Serializable]
    public class LinearComplianceFinder
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="LinearComplianceFinder"/> class.
        /// </summary>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="startYear">The value of the model year when modeling began.</param>
        /// <param name="endYear">The value of the model year when modeling ends.</param>
        /// <param name="settings">The modeling settings being used for the compliance modeling process.</param>
        /// <param name="logWriter">The <see cref="LogWriter"/> to use for logging the generated compliance findings.</param>
        public LinearComplianceFinder(Scenario scen, int startYear, int endYear, ModelingSettings settings, LogWriter logWriter)
        {
            this._scen        = scen;
            this._startYear   = startYear;
            this._endYear     = endYear;
            this._settings    = settings;
            this._logWriter   = logWriter;
            // techs and tech-groups
            this._techs       = settings.Technologies.TechnologyList;
            this._techGroups  = new List<TechnologyInfo>[9];
            for (int i = 0; i < this._techGroups.Length; i++)
            {
                this._techGroups[i] = new List<TechnologyInfo>();
            }
            for (int i = 0; i < this._techs.Length; i++)
            {
                int techIndex = this._techs[i].Index;
                //
                if (!TI.IsBaselineOnly(techIndex) && !TI.IsFCTimeBased(techIndex))
                {
                    if      (TI.IsBasicEnginePath       (techIndex) || // combine basic eng/turbo/adv-eng paths
                             TI.IsTurboEnginePath       (techIndex) ||
                             TI.IsAdvancedEnginePath    (techIndex)) { this._techGroups[ 0].Add(this._techs[i]); }
                    else if (TI.IsDieselEnginePath      (techIndex)) { this._techGroups[ 1].Add(this._techs[i]); }
                    else if (TI.IsManualTransmissionPath(techIndex)) { this._techGroups[ 2].Add(this._techs[i]); }
                    else if (TI.IsAutoTransmissionPath  (techIndex)) { this._techGroups[ 3].Add(this._techs[i]); }
                    else if (TI.IsElectrificationPath   (techIndex) || // combine electric/HEV/adv-HEV paths
                             TI.IsHybridElectricPath    (techIndex) ||
                             TI.IsAdvancedHybridElecPath(techIndex)) { this._techGroups[ 4].Add(this._techs[i]); }
                    else if (TI.IsDLRPath               (techIndex)) { this._techGroups[ 5].Add(this._techs[i]); }
                    else if (TI.IsROLLPath              (techIndex)) { this._techGroups[ 6].Add(this._techs[i]); }
                    else if (TI.IsMassReductionPath     (techIndex)) { this._techGroups[ 7].Add(this._techs[i]); }
                    else if (TI.IsAEROPath              (techIndex)) { this._techGroups[ 8].Add(this._techs[i]); }
                }
            }
            // init cf counters
            this._cfCount   = 0;
            //this._cfApplied = 0;

            // init runtime cf vars
            this._cfEff   = new ComplianceFinding();
            this._cfGroup = new ComplianceFinding();
            this._cfTmp   = new ComplianceFinding();
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Returns the next "most effective" <see cref="ComplianceFinding"/> for the scenario, model year, and manufacturer,
        /// based on the available runtime settings.
        /// </summary>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="modelYearData">An array of modeling data values, indexed by year, being analyzed with the current
        ///   scenario.  This array is populated with each subsequent model year.</param>
        /// <param name="mfrIndex">The index of the current manufacturer being analyzed.</param>
        /// <param name="overcomply">Specifies whether the model has entered over-compliance mode.</param>
        /// <param name="creditTrade">Specifies whether credit trading (transfers and carry fwd/back) is allowed.</param>
        /// <returns>The next valid <see cref="ComplianceFinding"/> using a linear, brute-force approach, or null, if the next
        ///   finding is not valid (e.g., ran out of solutions) or not efficient and 2nd best is not allowed.</returns>
        public ComplianceFinding GetNextFinding(ModelYear year, Industry[] modelYearData, int mfrIndex, bool overcomply, bool creditTrade)
        {
            this._cfEff.Clear();
            for (int i = 0; i < this._techGroups.Length; i++)
            {
                if (this._techGroups[i].Count == 0) { continue; }
                //
                this.FindComplianceFindingByGroup(this._techGroups[i], year, modelYearData, mfrIndex, overcomply, creditTrade);
                // obtain most efficient from current group's cf and overall cf
                ComplianceFinding eff = this.GetMostEfficient(this._cfEff, this._cfGroup);
                if (eff != this._cfEff) { this._cfEff.CopyFrom(eff); }
            }

            // post-process the finding:  determine if it's valid and/or efficient
            return (this._cfEff.IsEfficient || this._cfEff.IsValid) ? this._cfEff : null;
        }

        void FindComplianceFindingByGroup(List<TechnologyInfo> techGroup, ModelYear year, Industry[] modelYearData, int mfrIndex,
            bool overcomply, bool creditTrade)
        {
            // clear _cfGroup and populate the _cfTmp to use new params
            this._cfGroup.Clear();
            this._cfTmp.Initialize(this._scen, year, this._startYear, this._endYear, modelYearData, mfrIndex, this._settings,
                this._logWriter, techGroup);

            // initialize variables ...
            int          yrIndex = year.Index;
            Manufacturer mfr     = modelYearData[yrIndex].Manufacturers[mfrIndex];

            // generate component list based on the tech type --
            // the component list is a list of engines, transmissions, platforms, or vehicles
            IList componentList = null;
            if      (TI.IsEngineLevel      (techGroup[0].Index)) { componentList = mfr.Engines; }
            else if (TI.IsTransmissionLevel(techGroup[0].Index)) { componentList = mfr.Transmissions; }
            else if (TI.IsPlatformLevel    (techGroup[0].Index)) { componentList = mfr.Platforms; }
            else if (TI.IsVehicleLevel     (techGroup[0].Index)) { componentList = mfr.Vehicles; }
            else
            {
                ThrowHelper.InternalModelError();
            }

            // iterate each technology, filtering by tech-type
            for (int i = 0, techCount = techGroup.Count; i < techCount; i++)
            {   // process each technology
                //
                // examine i-th technology in group
                this.ExamineTechnology(techGroup, i, componentList, year, overcomply, creditTrade);

                // if found an efficient solution while operating in low-cost first mode, exit the tech-type loop
                // as there is no need to further examine more expensive solutions
                if (this._cfGroup.IsEfficient) { break; }
            } // next i (technology)
        }
        // when this method returns, _cfGroup contains the most efficient cf in the techType group
        void ExamineTechnology(List<TechnologyInfo> techGroup, int techIdx, IList componentList,
            ModelYear year, bool overcomply, bool creditTrade)
        {
            int t2Idx, t3Idx;    // for techs with 2-/3-way branches
            bool isMainBranch = this.FindTechBranches(techGroup, techGroup[techIdx].Index, out t2Idx, out t3Idx);
            if (isMainBranch)
            {   // the technology is the main branch -- examine each element in the componentList
                for (int i = 0, componentCount = componentList.Count; i < componentCount; i++)
                {   // obtain the component vehicles
                    List<Vehicle> vehs;
                    int vehIdx;     // -1 to examine all techs in vehs list (applicable to engs, trns, and aero tech types),
                                    // -or-, specific index to consider just one veh (applicable to all other tech types)
                    this.GenerateComponentVehicles(componentList, i, techGroup[techIdx], out vehs, out vehIdx);

                    // skip to the next component if there are no vehicles
                    if (vehs == null || vehs.Count == 0) { continue; }

                    // examine the component
                    this.ExamineComponent(techGroup, techIdx, vehs, vehIdx, t2Idx, t3Idx, year, overcomply, creditTrade);
                } // next i (component)
            } // end if (checking of main-branch)
        }
        // when this method returns, _cfGroup contains the most efficient cf from the techIdx, t2Idx, t3Idx, and t4Idx technologies
        void ExamineComponent(List<TechnologyInfo> techGroup, int techIdx, List<Vehicle> vehs, int vehIdx,
            int t2Idx, int t3Idx, ModelYear year, bool overcomply, bool creditTrade)
        {
            // generate compliance finding -- the most efficient for the current component
            this._cfTmp.ExamineFinding(techIdx, vehs, vehIdx, ref this._cfCount, overcomply, creditTrade);
            // obtain most efficient from current cf and current group's cf
            ComplianceFinding eff = this.GetMostEfficient(this._cfGroup, this._cfTmp);
            if (eff != this._cfGroup) { this._cfGroup.CopyFrom(eff); }

            if (t2Idx != -1)
            {   // 2-/3-/4-way tech branch -- look at the 2-nd, 3-rd, and/or 4-th choices
                this.ExamineComponent(techGroup, t2Idx, vehs, vehIdx, t3Idx, -1, year, overcomply, creditTrade);
            }
        }


        // ----- helper methods ----- //
        ComplianceFinding GetMostEfficient(ComplianceFinding cf1, ComplianceFinding cf2)
        {
            if      (!cf2.IsValid) { return cf1; }
            else if (!cf1.IsValid) { return cf2; }
            //else if (cf1.IsEfficient && cf2.IsEfficient)
            //{
            //    // both efficient: return the one with the most cost...
            //    if (cf1.TechCost < cf2.TechCost) { cf1.LogFinding(); return cf2; }
            //    else                             { cf2.LogFinding(); return cf1; }
            //}
            //else if (!cf2.IsEfficient && !cf2.IsEfficient)
            //{
            //    // both inefficient: return the one with the least cost...
            //    if (cf1.TechCost > cf2.TechCost) { cf1.LogFinding(); return cf2; }
            //    else                             { cf2.LogFinding(); return cf1; }
            //}
            else
            {   // log loosing CF and return the more efficient one
                if (cf2._efficiency <= cf1._efficiency) { if (!this._logWriter.IsEmpty) { cf1.LogFinding(); } return cf2; }
                else                                    { if (!this._logWriter.IsEmpty) { cf2.LogFinding(); } return cf1; }
            }
        }

        bool FindTechBranches(List<TechnologyInfo> techGroup, int techIdx, out int t2Idx, out int t3Idx)
        {
            t2Idx = t3Idx = -1;
            // look for tech decision points (e.g. AT6 vs DCT6)
            if      (techIdx == TI.DEAC  ) { t2Idx = this.FindTechInGroup(techGroup, TI.HCR   );
                                             t3Idx = this.FindTechInGroup(techGroup, TI.HCRP  ); }
            else if (techIdx == TI.AT6   ) { t2Idx = this.FindTechInGroup(techGroup, TI.DCT6  ); }
            else if (techIdx == TI.AT8   ) { t2Idx = this.FindTechInGroup(techGroup, TI.DCT8  );
                                             t3Idx = this.FindTechInGroup(techGroup, TI.CVT   ); }
            else if (techIdx == TI.BISG  ) { t2Idx = this.FindTechInGroup(techGroup, TI.CISG  ); }
            else if (techIdx == TI.SHEVP2) { t2Idx = this.FindTechInGroup(techGroup, TI.SHEVPS); }
            else if (techIdx == TI.BEV200) { t2Idx = this.FindTechInGroup(techGroup, TI.FCV   ); }
            // return false to skip 2-nd or 3-rd branch technologies (those will be examined with the main branch)
            else if (techIdx == TI.HCR || techIdx == TI.DCT6 || techIdx == TI.DCT8 || techIdx == TI.CVT ||
                     techIdx == TI.CISG || techIdx == TI.SHEVPS || techIdx == TI.FCV) { return false; }

            // return true to indicate the technology is a main branch
            return true;
        }
        int FindTechInGroup(List<TechnologyInfo> techGroup, int techIdx)
        {
            for (int i = 0, techCount = techGroup.Count; i < techCount; i++)
            {
                if (techGroup[i].Index == techIdx) { return i; }
            }
            return -1;
        }

        void GenerateComponentVehicles(IList componentList, int componentIndex, TechnologyInfo tech, out List<Vehicle> vehs, out int vehIdx)
        {
            object component = componentList[componentIndex];
            vehIdx = -1;
            vehs   = null;
            //
            if      (TI.IsEngineLevel      (tech.Index)) { vehs = ((Engine      )component).Vehicles; }
            else if (TI.IsTransmissionLevel(tech.Index)) { vehs = ((Transmission)component).Vehicles; }
            else if (TI.IsPlatformLevel    (tech.Index)) { vehs = ((Platform    )component).Vehicles; }
            else if (TI.IsVehicleLevel     (tech.Index)) { vehs = (List<Vehicle>)componentList; vehIdx = componentIndex; }
            else
            {
                ThrowHelper.InternalModelError();
            }
        }


        // ----- helper methods for swapping elements in arrays ----- //
        void Swap(Vehicle[] arr, int i, int j) { Vehicle tmp = arr[i]; arr[i] = arr[j]; arr[j] = tmp; }
        void Swap(ulong  [] arr, int i, int j) { ulong   tmp = arr[i]; arr[i] = arr[j]; arr[j] = tmp; }
        void Swap(int    [] arr, int i, int j) { int     tmp = arr[i]; arr[i] = arr[j]; arr[j] = tmp; }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the total count of <see cref="ComplianceFinding"/>s "examined" by this instance.</summary>
        public int CfCount { get { return this._cfCount; } }

        #endregion


        #region /*** Variables ***/

        Scenario         _scen;
        int              _startYear;
        int              _endYear;
        ModelingSettings _settings;
        LogWriter        _logWriter;
        // techs and tech-groups
        TechnologyInfo[]     _techs;
        List<TechnologyInfo>[] _techGroups;

        // --- cf counters --- //
        /// <summary>Represents the total number of Compliance Findings that have been analyzed thus far.</summary>
        int _cfCount;
        ///// <summary>Represents the total number of Compliance Findings that have been applied thus far.</summary>
        //int _cfApplied;

        // --- runtime CF variables --- //
        /// <summary>Specifies the most efficient <see cref="ComplianceFinding"/> thus far.</summary>
        ComplianceFinding _cfEff;
        /// <summary>Specifies the most efficient <see cref="ComplianceFinding"/> in a tech group currently being examined.</summary>
        ComplianceFinding _cfGroup;
        /// <summary>Specifies the temporary <see cref="ComplianceFinding"/>, based on current params (vehs/techs).</summary>
        ComplianceFinding _cfTmp;

        #endregion

    }
}
