#region << Using Directives >>
using System;
using System.Collections.Generic;
using Volpe.Cafe;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;
using TI = Volpe.Cafe.TechnologyIndexes;
using Volpe.Cafe.HelperClasses;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides static methods to assist the compliance modeling process in initializing and updating technology applicability.
    /// </summary>
    public sealed class TechnologyApplicability
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Static constructor to initialize superseding disallow-backfill technology arrays.
        /// </summary>
        static TechnologyApplicability()
        {
            TechSupersedes       = new int[TI.TechnologyCount][];
            TechSoftSupersedes   = new int[TI.TechnologyCount][];
            TechSupersedes2      = new int[TI.TechnologyCount][];
            TechDisallowBackfill = new int[TI.TechnologyCount][];
            TechRemoveForFCAdj   = new int[TI.TechnologyCount][];

            // set up tech-keys array
            TechKeys = KeyHelper.Keys;

            // set up reg-classes array
            RegClasses = new RegulatoryClass[] {RegulatoryClass.PassengerCar, RegulatoryClass.LightTruck, RegulatoryClass.LightTruck2b3};

            // ---------------------------------------------------------------------------
            // Initialize tech supersedes, disallow-backfill, and fc-adj-remove arrays.
            // ---------------------------------------------------------------------------
            // by default, initialize empty arrays for all techs
            // (the tech does not supersede anyone or disallows backfill)
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                TechSupersedes      [i] = new int[0];
                TechSoftSupersedes  [i] = new int[0];
                TechDisallowBackfill[i] = new int[0];
                TechRemoveForFCAdj  [i] = new int[0];
            }
            // ----- initialize basic engine technologies -----
            TechSupersedes      [TI.HCR     ] = new int[] { TI.VVL };
            TechSupersedes      [TI.HCRP    ] = new int[] { TI.VVL, TI.HCR };
            //
            TechDisallowBackfill[TI.DEAC    ] = new int[] { TI.HCR };
            TechDisallowBackfill[TI.HCR     ] = new int[] { TI.DEAC };
            TechDisallowBackfill[TI.HCRP    ] = new int[] { TI.DEAC };
            //
            TechRemoveForFCAdj  [TI.VVL     ] = new int[] { TI.VVT };
            TechRemoveForFCAdj  [TI.SGDI    ] = new int[] { TI.VVT, TI.VVL };
            TechRemoveForFCAdj  [TI.DEAC    ] = new int[] { TI.VVT, TI.VVL, TI.SGDI };
            TechRemoveForFCAdj  [TI.HCR     ] = new int[] { TI.VVT, TI.VVL, TI.SGDI };
            // ----- initialize turbo engine technologies -----
            TechSupersedes      [TI.TURBO1  ] = new int[] { TI.DEAC, TI.HCR, TI.HCRP };
            TechSupersedes      [TI.SEGR    ] = new int[] { TI.DEAC, TI.HCR, TI.HCRP, TI.TURBO1 };
            TechSupersedes      [TI.DWSP    ] = new int[] { TI.DEAC, TI.HCR, TI.HCRP, TI.TURBO1, TI.SEGR };
            TechSupersedes      [TI.TURBO2  ] = new int[] { TI.DEAC, TI.HCR, TI.HCRP, TI.TURBO1, TI.SEGR, TI.DWSP };
            TechSupersedes      [TI.CEGR1   ] = new int[] { TI.DEAC, TI.HCR, TI.HCRP, TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2 };
            TechSupersedes      [TI.CEGR1P  ] = new int[] { TI.DEAC, TI.HCR, TI.HCRP, TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1 };
            TechSupersedes      [TI.CEGR2   ] = new int[] { TI.DEAC, TI.HCR, TI.HCRP, TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR1P };
            //
            TechDisallowBackfill[TI.TURBO1  ] =
            TechDisallowBackfill[TI.SEGR    ] =
            TechDisallowBackfill[TI.DWSP    ] =
            TechDisallowBackfill[TI.TURBO2  ] =
            TechDisallowBackfill[TI.CEGR1   ] =
            TechDisallowBackfill[TI.CEGR1P  ] =
            TechDisallowBackfill[TI.CEGR2   ] = new int[] { TI.HCR, TI.HCRP, TI.HCR2, TI.CNG };
            //
            TechRemoveForFCAdj  [TI.TURBO1  ] = new int[] { TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR };
            TechRemoveForFCAdj  [TI.TURBO2  ] = new int[] { TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.TURBO1 };
            TechRemoveForFCAdj  [TI.CEGR1   ] = new int[] { TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.TURBO1, TI.TURBO2 };
            TechRemoveForFCAdj  [TI.CEGR2   ] = new int[] { TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.TURBO1, TI.TURBO2, TI.CEGR1 };
            // ----- initialize adv. engine technologies -----
            TechSupersedes      [TI.HCR2    ] = new int[] { TI.VVL, TI.HCR, TI.HCR2 };
            TechSupersedes      [TI.CNG     ] = new int[] { TI.LUBEFR1, TI.LUBEFR2, TI.LUBEFR3, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.HCRP };
            //
            TechDisallowBackfill[TI.HCR2    ] = new int[] { TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR1P, TI.CEGR2, TI.DEAC, TI.CNG };
            TechDisallowBackfill[TI.CNG     ] = new int[] { TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR1P, TI.CEGR2, TI.HCR, TI.HCRP, TI.HCR2 };
            //
            TechRemoveForFCAdj  [TI.CNG     ] = new int[] { TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR };
            // ----- initialize diesel engine technologies -----
            TechSupersedes      [TI.ADSL    ] =
            TechSupersedes      [TI.TURBODSL] =
            TechSupersedes      [TI.DWSPDSL ] =
            TechSupersedes      [TI.EFRDSL  ] =
            TechSupersedes      [TI.CLCDSL  ] =
            TechSupersedes      [TI.LPEGRDSL] =
            TechSupersedes      [TI.DSIZEDSL] = new int[] { TI.LUBEFR1, TI.LUBEFR2, TI.LUBEFR3, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.HCRP };
            //
            TechRemoveForFCAdj  [TI.ADSL    ] = new int[] { TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR };
            // ----- initialize manual transmission technologies -----
            TechSupersedes      [TI.MT6     ] = new int[] { TI.MT5 };
            TechSupersedes      [TI.MT7     ] = new int[] { TI.MT5, TI.MT6 };
            //
            TechRemoveForFCAdj  [TI.MT6     ] = new int[] { TI.MT5 };
            TechRemoveForFCAdj  [TI.MT7     ] = new int[] { TI.MT5, TI.MT6 };
            // ----- initialize automatic transmission technologies -----
            TechSupersedes      [TI.AT6     ] = new int[] { TI.AT5 };
            TechSupersedes      [TI.AT6P    ] = new int[] { TI.AT5, TI.AT6 };
            TechSupersedes      [TI.AT8     ] = new int[] { TI.AT5, TI.AT6, TI.AT6P };
            TechSupersedes      [TI.AT8P    ] = new int[] { TI.AT5, TI.AT6, TI.AT6P, TI.AT8 };
            TechSupersedes      [TI.DCT6    ] = new int[] { TI.TATI, TI.AT5 };
            TechSupersedes      [TI.DCT8    ] = new int[] { TI.TATI, TI.AT5, TI.DCT6 };
            TechSupersedes      [TI.CVT     ] = new int[] { TI.TATI, TI.AT5, TI.AT6, TI.AT6P };
            //
            TechDisallowBackfill[TI.AT6     ] = new int[] { TI.DCT6, TI.DCT8 };
            TechDisallowBackfill[TI.AT6P    ] = new int[] { TI.DCT6, TI.DCT8 };
            TechDisallowBackfill[TI.AT8     ] = new int[] { TI.DCT6, TI.DCT8, TI.CVT };
            TechDisallowBackfill[TI.AT8P    ] = new int[] { TI.DCT6, TI.DCT8, TI.CVT };
            TechDisallowBackfill[TI.DCT6    ] =
            TechDisallowBackfill[TI.DCT8    ] = new int[] { TI.AT6 , TI.AT6P, TI.AT8, TI.AT8P, TI.CVT };
            TechDisallowBackfill[TI.CVT     ] = new int[] { TI.DCT6, TI.DCT8, TI.AT8, TI.AT8P };
            //
            TechRemoveForFCAdj  [TI.AT6     ] = new int[] { TI.AT5 };
            TechRemoveForFCAdj  [TI.AT8     ] = new int[] { TI.AT5, TI.AT6 };
            TechRemoveForFCAdj  [TI.DCT6    ] = new int[] { TI.AT5 };
            TechRemoveForFCAdj  [TI.DCT8    ] = new int[] { TI.AT5, TI.DCT6 };
            TechRemoveForFCAdj  [TI.CVT     ] = new int[] { TI.AT5, TI.AT6 };
            // ----- initialize electrification technologies -----
            TechSupersedes      [TI.BISG    ] = new int[] { TI.SS12V };
            TechSupersedes      [TI.CISG    ] = new int[] { TI.SS12V };
            //
            TechDisallowBackfill[TI.BISG    ] = new int[] { TI.CISG };
            TechDisallowBackfill[TI.CISG    ] = new int[] { TI.BISG };
            //
            TechRemoveForFCAdj  [TI.BISG    ] = new int[] { TI.SS12V };
            TechRemoveForFCAdj  [TI.CISG    ] = new int[] { TI.SS12V };
            // ----- initialize hybrid technologies -----
            TechSupersedes      [TI.SHEVP2  ] = new int[] {
                                                // transmission technologies --
                                                // | for P2, do not automatically supersede transmission type, otherwise
                                                // | accounting issues arise; instead, supersede all trn types except DCT6
                                                // | when P2 is applied to a vehicle during modeling
                                                TI.TATI,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG };
            TechSupersedes      [TI.SHEVPS  ] = new int[] {
                                                // engine technologies
                                                TI.TEFRI, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.HCRP,
                                                // advanced engine technologies
                                                TI.HCR2, TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR1P, TI.CEGR2,
                                                // transmission technologies
                                                TI.TATI, TI.AT5, TI.AT6, TI.AT6P, TI.AT8, TI.AT8P, TI.CVT, TI.DCT6, TI.DCT8,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG };
            //
            TechDisallowBackfill[TI.SHEVP2  ] = new int[] { TI.CISG, TI.SHEVPS };
            TechDisallowBackfill[TI.SHEVPS  ] = new int[] { TI.BISG, TI.SHEVP2 };
            //
            TechRemoveForFCAdj  [TI.SHEVP2  ] = new int[] {
                                                // transmission technologies
                                                TI.AT5, TI.AT6, TI.AT8, TI.DCT6, TI.DCT8, TI.CVT,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG };
            TechRemoveForFCAdj  [TI.SHEVPS  ] = new int[] {
                                                // engine technologies
                                                TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR,
                                                // advanced engine technologies
                                                TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.TURBO2, TI.CEGR1, TI.CEGR2,
                                                // transmission technologies
                                                TI.AT5, TI.AT6, TI.AT8, TI.DCT6, TI.DCT8, TI.CVT,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG };
            // ----- initialize adv. hybrid technologies -----
            TechSupersedes      [TI.PHEV30  ] = new int[] {
                                                // engine technologies
                                                TI.TEFRI, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.HCRP,
                                                // advanced engine technologies
                                                TI.HCR2, TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR1P, TI.CEGR2,
                                                // transmission technologies
                                                TI.TATI, TI.AT5, TI.AT6, TI.AT6P, TI.AT8, TI.AT8P, TI.CVT, TI.DCT6, TI.DCT8,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG, TI.SHEVP2, TI.SHEVPS };
            TechSupersedes      [TI.PHEV50  ] = new int[] {
                                                // engine technologies
                                                TI.TEFRI, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.HCRP,
                                                // advanced engine technologies
                                                TI.HCR2, TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR1P, TI.CEGR2,
                                                // transmission technologies
                                                TI.TATI, TI.AT5, TI.AT6, TI.AT6P, TI.AT8, TI.AT8P, TI.CVT, TI.DCT6, TI.DCT8,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG, TI.SHEVP2, TI.SHEVPS, TI.PHEV30 };
            TechSupersedes      [TI.BEV200  ] =
            TechSupersedes      [TI.FCV     ] = new int[] {
                                                // EV & FCV remove the engine and supersede all eng techs
                                                TI.TEFRI, TI.SOHC, TI.DOHC, TI.OHV, TI.LUBEFR1, TI.LUBEFR2, TI.LUBEFR3,
                                                TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR, TI.HCRP,
                                                // advanced engine technologies
                                                TI.HCR2, TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR1P, TI.CEGR2,
                                                // EV & FCV remove the transmission and supersede all trn techs
                                                TI.TATI, TI.AT5, TI.AT6, TI.AT6P, TI.AT8, TI.AT8P, TI.CVT, TI.DCT6, TI.DCT8,
                                                // also supersede preceding electrification techs
                                                TI.SS12V, TI.BISG, TI.CISG, TI.SHEVP2, TI.SHEVPS, TI.PHEV30, TI.PHEV50 };
            //
            TechDisallowBackfill[TI.BEV200  ] = new int[] { TI.FCV };
            TechDisallowBackfill[TI.FCV     ] = new int[] { TI.BEV200 };
            //
            TechRemoveForFCAdj  [TI.PHEV30  ] = new int[] {
                                                // engine technologies
                                                TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR,
                                                // advanced engine technologies
                                                TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.TURBO2, TI.CEGR1, TI.CEGR2,
                                                // transmission technologies
                                                TI.AT5, TI.AT6, TI.AT8, TI.DCT6, TI.DCT8, TI.CVT,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG,
                                                // hybrid technologies
                                                TI.SHEVP2, TI.SHEVPS };
            TechRemoveForFCAdj[TI.PHEV50] = new int[] {
                                                // engine technologies
                                                TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR,
                                                // advanced engine technologies
                                                TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.TURBO2, TI.CEGR1, TI.CEGR2,
                                                // transmission technologies
                                                TI.AT5, TI.AT6, TI.AT8, TI.DCT6, TI.DCT8, TI.CVT,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG,
                                                // hybrid technologies
                                                TI.SHEVP2, TI.SHEVPS, TI.PHEV30 };
            TechRemoveForFCAdj  [TI.BEV200  ] =
            TechRemoveForFCAdj  [TI.FCV     ] = new int[] {
                                                // engine technologies
                                                TI.SOHC, TI.DOHC, TI.OHV, TI.VVT, TI.VVL, TI.SGDI, TI.DEAC, TI.HCR,
                                                // advanced engine technologies
                                                TI.CNG,
                                                // turbo engine technologies
                                                TI.TURBO1, TI.TURBO2, TI.CEGR1, TI.CEGR2,
                                                // transmission technologies
                                                TI.AT5, TI.AT6, TI.AT8, TI.DCT6, TI.DCT8, TI.CVT,
                                                // electrification technologies
                                                TI.SS12V, TI.BISG, TI.CISG,
                                                // hybrid technologies
                                                TI.SHEVP2, TI.SHEVPS, TI.PHEV30, TI.PHEV50 };
            // ----- initialize low rolling resistance tires technologies -----
            TechRemoveForFCAdj  [TI.ROLL20  ] = new int[] { TI.ROLL10 };
            // ----- initialize mass reduction technologies -----
            //TechSoftSupersedes[TI.MR2     ] =
            //TechSoftSupersedes[TI.MR3     ] =
            //TechSoftSupersedes[TI.MR4     ] =
            //TechSoftSupersedes[TI.MR5     ] = new int[] { TI.TURBO1, TI.SEGR, TI.DWSP, TI.TURBO2, TI.CEGR1, TI.CEGR2, TI.MILLER };
            //
            TechRemoveForFCAdj  [TI.MR2     ] = new int[] { TI.MR1 };
            TechRemoveForFCAdj  [TI.MR3     ] = new int[] { TI.MR1, TI.MR2 };
            TechRemoveForFCAdj  [TI.MR4     ] = new int[] { TI.MR1, TI.MR2, TI.MR3 };
            TechRemoveForFCAdj  [TI.MR5     ] = new int[] { TI.MR1, TI.MR2, TI.MR3, TI.MR4 };
            // ----- initialize aerodynamic improvements technologies -----
            TechRemoveForFCAdj  [TI.AERO20  ] = new int[] { TI.AERO10 };

            // combine TechSupersedes and TechSoftSupersedes arrays
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                TechSupersedes2[i] = new int[TechSupersedes[i].Length + TechSoftSupersedes[i].Length];
                //
                Array.Copy(TechSupersedes    [i], 0, TechSupersedes2[i],                        0, TechSupersedes    [i].Length);
                Array.Copy(TechSoftSupersedes[i], 0, TechSupersedes2[i], TechSupersedes[i].Length, TechSoftSupersedes[i].Length);
            } // next i (technology)
        }

        #endregion

        #region /*** Methods ***/

        #region /* Technology Initialization, Enabling, and Verification */

        /// <summary>
        /// Initializes the technology availability and usage on a <see cref="Component"/> for the modeling process.
        /// </summary>
        /// <param name="component">The <see cref="Component"/>, whose technology availability and usage to initialize.</param>
        /// <param name="settings">The modeling settings to use for initialization.</param>
        public static void Initialize(Component component, ModelingSettings settings)
        {
            // initialize technology applicability arrays
            Component.CModelData cmd = component.ModelData;
            //
            cmd.TechUsed           = new bool[TI.TechnologyCount];
            cmd.TechApplied        = new bool[TI.TechnologyCount];
            cmd.TechAppliedYear    = new int [TI.TechnologyCount];
            cmd.TechSuperseded     = new bool[TI.TechnologyCount];
            cmd.TechSupersededYear = new int [TI.TechnologyCount];
            // set usage based on platform leader
            if (component.Vehicles.Count > 0)
            {
                Vehicle veh      = component.GetPlatformLeader();
                int[]   techList = component.TechnologyList;
                for (int i = 0; i < techList.Length; i++)
                {
                    int techIndex = techList[i];
                    if (veh.ModelData.TechUsed[techIndex])
                    {
                        cmd.TechUsed[techIndex] = true;
                    }
                }
            }
            // update baseline component version
            component.UpdateComponentVersion(null, null);
        }

        /// <summary>
        /// Initializes the technology availability and usage for the modeling process.
        /// </summary>
        /// <param name="vehicle">The vehicle, whose technology availability and usage to initialize.</param>
        /// <param name="settings">The modeling settings to use for initialization.</param>
        public static void Initialize(Vehicle vehicle, ModelingSettings settings)
        {
            //----------------------------------------------------------------------------------//
            // Initializes the general technology availability and initial usage based on       //
            // technology path, overrides, and vehicle, engine, and transmission specification. //
            //                                                                                  //
            // At the end of initialization, RectifyDependencies is called to verify            //
            // consistency among various technologies, and to ensure that technologies marked   //
            // as used properly disable other conflicting technologies.  RectifyDependencies    //
            // also turns off availability of all technologies marked as used during            //
            // initialization, thus it is not necessary to do so here.                          //
            //----------------------------------------------------------------------------------//

            // get vehicle, engine, and transmission description
            bool                      hasEng = (vehicle.Engine       != null);
            bool                      hasTrn = (vehicle.Transmission != null);
            Vehicle     .CDescription vd     = vehicle.Description;
            Engine      .CDescription ed     = (hasEng) ? vehicle.Engine      .Description : null;
            Transmission.CDescription td     = (hasTrn) ? vehicle.Transmission.Description : null;
            Vehicle     .CModelData   vmd    = vehicle.ModelData;   // get vehicle modeling data

            // get tech-class and technologies info
            TechnologyInfo[] technologies = settings.Technologies.TechnologyList;

            // initialize technology applicability arrays
            vmd.TechAvailable         = new bool  [TI.TechnologyCount];
            vmd.TechUsed              = new bool  [TI.TechnologyCount];
            vmd.TechInherited         = new bool  [TI.TechnologyCount];
            vmd.TechApplied           = new bool  [TI.TechnologyCount];
            vmd.TechAppliedYear       = new int   [TI.TechnologyCount];
            vmd.TechSuperseded        = new bool  [TI.TechnologyCount];
            vmd.TechSupersededYear    = new int   [TI.TechnologyCount];
            //
            vmd.TechEnabled           = new bool  [TI.TechnologyCount];
            vmd.TechIgnored           = new bool  [TI.TechnologyCount];
            //
            vmd.TechCostTimeBased     = new double[TI.TechnologyCount];

            // set TechUsed & TechAvailable based on the vehicle, engine, and transmission description
            for (int i = 0; i < vd.UsedTechnologies.Length; i++) { vmd.TechUsed[vd.UsedTechnologies[i]] = true; }
            for (int i = 0; i < vd.AvailableTechnologies.Length; i++)
            {
                int index = vd.AvailableTechnologies[i];
                vmd.TechAvailable[index] = technologies[index].GetAttributes(vehicle).Applicable;
            }

            if (hasEng)
            {
                for (int i = 0; i < ed.UsedTechnologies.Length; i++) { vmd.TechUsed[ed.UsedTechnologies[i]] = true; }
                for (int i = 0; i < ed.AvailableTechnologies.Length; i++)
                {
                    int index = ed.AvailableTechnologies[i];
                    vmd.TechAvailable[index] = technologies[index].GetAttributes(vehicle).Applicable;
                }
            }
            if (hasTrn)
            {
                for (int i = 0; i < td.UsedTechnologies.Length; i++) { vmd.TechUsed[td.UsedTechnologies[i]] = true; }
                for (int i = 0; i < td.AvailableTechnologies.Length; i++)
                {
                    int index = td.AvailableTechnologies[i];
                    vmd.TechAvailable[index] = technologies[index].GetAttributes(vehicle).Applicable;
                }
            }

            // set initial glider weight for the vehicle
            double gliderWeight = vd.CurbWeight * GliderShare[vehicle.TechnologyClass];
            double deltaWeightPCT = 0;
            for (int i = 0; i < technologies.Length; i++)
            {
                if (TI.IsMassReductionPath(i) && vmd.TechUsed[i])
                {
                    deltaWeightPCT += technologies[i].GetAttributes(vehicle).DeltaWeightPCT;
                }
            }
            vd.GliderWeightMR0 = gliderWeight / (1 - deltaWeightPCT);

            // calculate initial off-cycle credit for the vehicle
            vmd.OffCycleCredits = 0;
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                if (vmd.TechUsed[i])
                {
                    vmd.OffCycleCredits += technologies[i].OffCycleCredit[vehicle.RegClass];
                }
            }

            // check used states to update certain vehicle flags
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                if (vmd.TechUsed[i])
                {
                    if      (TI.IsConversionToMHEV(i)) { vehicle.HEVType = HEVType.MildHybrid  ; }
                    else if (TI.IsConversionToSHEV(i)) { vehicle.HEVType = HEVType.StrongHybrid; }
                    else if (TI.IsConversionToPHEV(i)) { vehicle.HEVType = HEVType.PlugInHybrid; }
                    else if (TI.IsConversionToEV  (i)) { vehicle.HEVType = HEVType.PureElectric; }
                    else if (TI.IsConversionToFCV (i)) { vehicle.HEVType = HEVType.FuelCell    ; }
                    //
                    if (TI.IsConversionToPHEV(i) || TI.IsConversionToEV(i) || TI.IsConversionToFCV(i))
                    {
                        vehicle.ModelData.ZEVCredits = technologies[i].ZEVCredits;
                    }
                }
            }

            // update superseded states for techs set to USED in the baseline
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                if (vmd.TechUsed[i])
                {
                    int[] supersededTechs = TechSupersedes[i];
                    for (int j = 0; j < supersededTechs.Length; j++)
                    {
                        if (vmd.TechUsed[supersededTechs[j]])
                        {   // this technology is set to USED in the baseline, but another technology
                            // that supersedes this one is USED as well -- set this one as superseded
                            vmd.TechSuperseded[supersededTechs[j]] = true;
                        }
                    }
                }
            }

            // remove engine and transmission for hybrid vehicles
            //  - SHEVP2, remove transmission
            //  - SHEVPS, remove engine and transmission
            //  - PHEV/EV/FCV, remove engine and transmission
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                if (vmd.TechUsed[i])
                {
                    if (TI.IsConversionToSHEV(i) || TI.IsConversionToPHEV(i) ||
                        TI.IsConversionToEV  (i) || TI.IsConversionToFCV (i))
                    {
                        if (i != TI.SHEVP2) { vehicle.RemoveEngine(); }
                        vehicle.RemoveTransmission();
                    }
                }
            }

            // ---------------------------------------------------------------------------------//
            // rectify engine, transmission, and other (vehicle) dependencies                   //
            // ---------------------------------------------------------------------------------//
            TechnologyApplicability.RectifyDependencies(vehicle, null, -1, settings);
        }

        /// <summary>
        /// Sets technology enable state for the specified model year for the modeling process.
        /// </summary>
        /// <param name="vehicle">The vehicle, whose technology enable state to update.</param>
        /// <param name="year">The model year for which to update technology enabled states.</param>
        /// <param name="techIndex">The index of the technology to update, or -1 if updating all technologies.  When this method
        ///   is called during model year pre-processing, specify -1 to examine all technologies.  When this method is called
        ///   after a technology has been installed on a vehicle, specify the index of the technology to examine.</param>
        /// <param name="settings">The modeling settings to use for enabling technology states.</param>
        public static void Enable(Vehicle vehicle, ModelYear year, int techIndex, ModelingSettings settings)
        {
            //----------------------------------------------------------------------------------//
            // Enables the technology availability for a specific model year based on general   //
            // availability, the year technology becomes available, and redesign and refresh    //
            // years.  A single technology at a time, or all of the technologies at once can be //
            // enabled.  At the start of the model year, all of the technologies must be        //
            // enabled; during the compliance modeling process, only the technologies that were //
            // installed on a vehicle must be enabled (or "enable updated").                    //
            //                                                                                  //
            // At the end of enabling, RectifyDependencies is called to verify consistency      //
            // among various technologies, and to ensure that technologies marked as used       //
            // properly disable other conflicting technologies.  RectifyDependencies also turns //
            // off enabled states of all technologies marked as used during enabling, thus it   //
            // is not necessary to do so here.                                                  //
            //----------------------------------------------------------------------------------//

            // set up variables
            Vehicle.CModelData vmd          = vehicle.ModelData;
            TechnologyInfo[]   technologies = settings.Technologies.TechnologyList;

            // perform base initialization of tech-enabled array (and optionally, tech-ignored array)
            if (techIndex == -1)
            {   // examing all techs -- initialize technology enabled and ignored arrays
                for (int i = 0; i < TI.TechnologyCount; i++)
                {
                    TechnologyAttributes attr = technologies[i].GetAttributes(vehicle);
                    //
                    vmd.TechEnabled [i] = vmd.TechAvailable[i] &&
                        (settings.OperatingModes.IgnoreYearAvailable || attr.YearAvailable <= year.Year) &&
                        (attr.YearRetired == 0 || attr.YearRetired >= year.Year);
                    vmd.TechIgnored [i] = false;
                }
            }
            else
            {   // examining one tech -- initialize technology enabled for techIndex only
                TechnologyAttributes attr = technologies[techIndex].GetAttributes(vehicle);
                //
                vmd.TechEnabled[techIndex] = vmd.TechAvailable[techIndex] &&
                    (attr.YearAvailable <= year.Year) &&
                    (attr.YearRetired == 0 || attr.YearRetired >= year.Year);
            }

            // computes technology start and end ranges, based on whether examining one or all techs
            int sIdx = (techIndex == -1) ? 0                  : techIndex;
            int eIdx = (techIndex == -1) ? TI.TechnologyCount : techIndex + 1;

            // if considering refresh/redesign schedules, check vehicle-specific refresh/redesign years
            if (!settings.OperatingModes.IgnoreRefreshRedesign)
            {   //
                // get redesign/refresh states
                bool atRedesign = vehicle.IsAtRedesign(year);
                bool atRefresh  = vehicle.IsAtRefresh (year);
                // iterate each technology
                for (int i = sIdx; i < eIdx; i++)
                {   // check for redesign/refresh
                    if (vmd.TechEnabled[i])
                    {   // disable redesign-level technologies if the vehicle is not at redesign;
                        // disable refresh-level technologies if the vehicle is not at redesign or refresh
                        if ((!atRedesign && TI.IsTiedToRedesign(i)) || (!atRedesign && !atRefresh && TI.IsTiedToRefresh(i)))
                        {
                            vmd.TechEnabled[i] = false;
                        }
                    } // end if (tech enabled for shadow of redesign/refresh)
                } // next i (tech)
            }

            // ---------------------------------------------------------------------------------//
            // Rectify engine, transmission, and other (vehicle) dependencies                   //
            // ---------------------------------------------------------------------------------//
            TechnologyApplicability.RectifyDependencies(vehicle, year, techIndex, settings);
        }

        /// <summary>
        /// Rectifies the usage with availability/enabled dependencies amongst various technology lines.
        /// </summary>
        /// <param name="vehicle">The vehicle being affected.</param>
        /// <param name="year">The model year for which to rectify dependencies, or null, if not applicable.  If a model year is
        ///   specified, the vehicle's TechEnabled setting is updated; if model year is null, the vehicle's TechAvailable setting
        ///   is updated.</param>
        /// <param name="techIdx">The index of the technology to update, or -1 if updating all technologies.</param>
        /// <param name="settings">The modeling settings to use for enabling technology states, where required.</param>
        static void RectifyDependencies(Vehicle vehicle, ModelYear year, int techIdx, ModelingSettings settings)
        {
            //----------------------------------------------------------------------------------//
            // -- initially, all techs that can be enabled are set as enabled (e.g., enabled,   //
            //    if path and year available permits)                                           //
            // -- this method goes through and disables certain technologies based on the       //
            //    following criteria:                                                           //
            //     1).  disable if a technology depends on another technology being used, while //
            //          the other tech is not used                                              //
            //     2).  disable if a technology conflicts with another technology, when another //
            //          tech is used                                                            //
            //----------------------------------------------------------------------------------//

            Vehicle  .CModelData vmd =  vehicle.ModelData; // get references to veh/eng/trn/platform modeling data
            Component.CModelData emd = (vehicle.Engine       == null) ? null : vehicle.Engine      .ModelData;
            Component.CModelData tmd = (vehicle.Transmission == null) ? null : vehicle.Transmission.ModelData;
            Component.CModelData pmd = (vehicle.Platform     == null) ? null : vehicle.Platform    .ModelData;
            bool   updateTechEnabled = (year != null);
            bool    updateSuperseded = (techIdx != -1);
            bool[]            techAE = (updateTechEnabled) ? vmd.TechEnabled : vmd.TechAvailable;
            bool[]          techUsed = vmd.TechUsed;

            int sIdx = (techIdx == -1) ? 0                  : techIdx;
            int eIdx = (techIdx == -1) ? TI.TechnologyCount : techIdx + 1;

            bool afterBasicEngRectified = false; // whether dependencies after basic engs have been rectified
            bool afterShevRectified     = false; // whether dependencies after SHEV have been rectified

            for (int i = sIdx; i < eIdx; i++)
            {   // disable available/enabled state for this tech if it is already used
                if (techUsed[i]) { techAE[i] = false; }

                // rectify dependencies ...
                RectifyDependenciesHelper(i, techUsed, techAE, updateTechEnabled, ref afterBasicEngRectified, ref afterShevRectified,
                    vehicle, year, techIdx, settings);
 
                //------------------------------------------------------------------------------//
                // rectify superseded technology dependencies ...                               //
                //------------------------------------------------------------------------------//
                if (updateSuperseded)
                {
                    RectifyDependencies_UpdateSuperseded(i, vmd);
                    RectifyDependencies_UpdateSuperseded(i, emd);
                    RectifyDependencies_UpdateSuperseded(i, tmd);
                    RectifyDependencies_UpdateSuperseded(i, pmd);
                }
            } // next i (technology)
        }
        static void RectifyDependencies_UpdateSuperseded(int techIndex, Vehicle.CModelData vmd)
        {
            RectifyDependencies_UpdateSuperseded(techIndex, vmd.TechUsed, vmd.TechApplied, vmd.TechAppliedYear, vmd.TechSuperseded, vmd.TechSupersededYear);
        }
        static void RectifyDependencies_UpdateSuperseded(int techIndex, Component.CModelData cmd)
        {
            if (cmd != null)
            {
                RectifyDependencies_UpdateSuperseded(techIndex,
                    cmd.TechUsed, cmd.TechApplied, cmd.TechAppliedYear, cmd.TechSuperseded, cmd.TechSupersededYear);
            }
        }
        static void RectifyDependencies_UpdateSuperseded(int techIndex,
            bool[] techUsed, bool[] techApplied, int[] techAppliedYear, bool[] techSuperseded, int[] techSupersededYear)
        {
            if (!techUsed[techIndex] || !techApplied[techIndex]) { return; }

            // examine all tech supersedes indexes of the current tech
            int[] techSupersedes = TechSupersedes[techIndex];
            for (int j = 0; j < techSupersedes.Length; j++)
            {   // the tech can be marked as superseded only if it is already used, but has not been superseded yet
                int stIndex = techSupersedes[j];
                if (techUsed[stIndex] && !techSuperseded[stIndex])
                {
                    techSuperseded    [stIndex] = true;
                    techSupersededYear[stIndex] = techAppliedYear[techIndex];
                }
            }
            // also examine soft-supersedes indexes of the current veh
            int[] techSoftSupersedes = TechSoftSupersedes[techIndex];
            for (int j = 0; j < techSoftSupersedes.Length; j++)
            {   // the tech can be marked as soft-superseded only if it is already used, but has not been soft-superseded yet
                if (techUsed[techSoftSupersedes[j]] && techSupersededYear[techSoftSupersedes[j]] == 0)
                {   // for soft-supersedes, only consider if the superseding technology was not applied in the same year
                    // as the technology being superseded
                    if (techAppliedYear[techIndex] > techAppliedYear[techSoftSupersedes[j]])
                    {
                        techSupersededYear[techSoftSupersedes[j]] = techAppliedYear[techIndex];
                    }
                }
            }
        }
        static void RectifyDependenciesHelper(int i, bool[] techUsed, bool[] techArr, bool updateTechEnabled,
            ref bool afterBasicEngRectified, ref bool afterShevRectified,
            Vehicle vehicle, ModelYear year, int techIdx, ModelingSettings settings)
        {
            //
            // --------------------------------------------------
            // rectify engine technology dependencies
            // --------------------------------------------------
            //
            bool[] techAvailable = vehicle.ModelData.TechAvailable;
            // --- basic engine path ---
            if (TI.IsBasicEnginePath(i))
            {
                if (techUsed[i])
                {
                    if (settings.OperatingModes.BackfillMissingBEP)
                    {   // enforce consistency within the basic engine technology group
                        //
                        // DEAC and HCR are mutually exclusive -- if either one is used, disable the other
                        // if DEAC is used, also disable HCRP and HCR2
                        // if HCR is used, also disable all preceding technologies since those are prerequisites
                        if (i == TI.DEAC)
                        {
                            techArr[TI.HCR ] = false;
                            techArr[TI.HCRP] = false;
                            techArr[TI.HCR2] = false;
                        }
                        else if (i == TI.HCR || i == TI.HCRP)
                        {
                            techArr[TI.VVT ] = false;
                            techArr[TI.VVL ] = false;
                            techArr[TI.SGDI] = false;
                            techArr[TI.DEAC] = false;
                            techArr[TI.HCR ] = false; // disable HCR if HCRP is used
                        }
                    }
                    else
                    {   // enforce linearity of basic engine technologies
                        //
                        // if VVL or SGDI is used, disable VVT and VVL
                        if (i == TI.VVL || i == TI.SGDI)
                        {
                            techArr[TI.VVT] = false;
                            techArr[TI.VVL] = false;
                        }
                        // DEAC and HCR are mutually exclusive -- if either one is used, disable both
                        // also disable all preceding technologies since those are prerequisites
                        // if DEAC is used, also disable HCRP and HCR2
                        if (i == TI.DEAC || i == TI.HCR || i == TI.HCRP)
                        {
                            techArr[TI.VVT ] = false;
                            techArr[TI.VVL ] = false;
                            techArr[TI.SGDI] = false;
                            techArr[TI.DEAC] = false;
                            techArr[TI.HCR ] = false;
                            //
                            if (i == TI.DEAC)
                            {
                                techArr[TI.HCRP] = false;
                                techArr[TI.HCR2] = false;
                            }
                        }
                    }
                    // HCR can only be applied to DOHC engines
                    if (i == TI.SOHC || i == TI.OHV)
                    {
                        techArr[TI.HCR ] = false;
                        techArr[TI.HCRP] = false;
                        techArr[TI.HCR2] = false;
                    }
                }
                // rectify basic engine path dependencies (decides whether to enable TURBO/adv-Eng/DSL/HEV paths if
                // basic eng path was satisfied)
                if (updateTechEnabled && !afterBasicEngRectified)
                {
                    RectifyDependencies_EnableAfterBasicEng(techArr, techUsed, vehicle, year, techIdx, settings);
                    afterBasicEngRectified = true; // to prevent redundant checks when examining multiple techs
                }
            }
            // --- turbo engine techs ---
            else if (TI.IsTurboEnginePath(i))
            {
                if (techUsed[i])
                {   // basic engine path cannot be applied once any of turbo-eng become used
                    //  -AND-
                    // turbo-eng, adv-eng, dsl-eng, and shev are mutually exclusive (except SHEVP2)
                    // however, allow conversion from trubo path to hev/ev/fcv path
                    RectifyDependencies_DisableTechPaths(techAvailable, techArr, true, false, true, true, false, false, false, false, false, false, false, false, false);
                }
            }
            else if (TI.IsAdvancedEnginePath(i))
            {
                if (techUsed[i])
                {   // basic engine path cannot be applied once any of adv-eng become used
                    //  -AND-
                    // turbo-eng, adv-eng, dsl-eng, and shev are mutually exclusive (except SHEVP2)
                    // however, allow conversion from adv-eng path to hev/ev/fcv path (only for HCR2)
                    bool isHCR2 = (i == TI.HCR2);
                    RectifyDependencies_DisableTechPaths(techAvailable, techArr, true, true, false, true, false, false, false, !isHCR2, !isHCR2, false, false, false, false);
                }
            }
            else if (TI.IsDieselEnginePath(i))
            {
                if (techUsed[i])
                {   // basic engine path cannot be applied once any of dsl-eng become used
                    //  -AND-
                    // turbo-eng, adv-eng, dsl-eng, and shev are mutually exclusive
                    RectifyDependencies_DisableTechPaths(techAvailable, techArr, true, true, true, false, false, false, false, true, true, false, false, false, false);
                }
            }
            //
            // --------------------------------------------------
            // rectify transmission technology dependencies
            // --------------------------------------------------
            //
            // --- manual transmission path ---
            else if (TI.IsManualTransmissionPath(i))
            {
                if (techUsed[i])
                {   // if any of manual transmissions are used, disable automatic transmission path and strong hybrids
                    RectifyDependencies_DisableTechPaths(techAvailable, techArr, false, false, false, false, false, true, false, true, true, false, false, false, false);
                    // also disable micro and mild HEVs
                    techArr[TI.SS12V] = false;
                    techArr[TI.BISG ] = false;
                    techArr[TI.CISG ] = false;
                }
            }
            // --- automatic transmission path ---
            else if (TI.IsAutoTransmissionPath(i))
            {
                if (techUsed[i])
                {   // if any of automatic transmissions are used, disable manual transmission path
                    RectifyDependencies_DisableTechPaths(techAvailable, techArr, false, false, false, false, true, false, false, false, false, false, false, false, false);

                    //---------------------------------------------------------------------------
                    // additionally, special logic is required for branches along the automatic
                    // transmission path -- the decision tree looks as follows:
                    //
                    //         AT5
                    //        /   \
                    //     AT6     DCT6
                    //      |       |
                    //     AT6P    DCT8
                    //     /  \
                    //   AT8  CVT
                    //    |
                    //   AT6
                    //
                    // where:
                    //  AT/CVT and DCT sub-paths are mutually exclusive
                    //  CVT and AT8 technologies are mutually exclusive
                    //---------------------------------------------------------------------------
                    if (i == TI.DCT6 || i == TI.DCT8)
                    {
                        techArr[TI.AT6 ] = false;
                        techArr[TI.AT6P] = false;
                        techArr[TI.AT8 ] = false;
                        techArr[TI.AT8P] = false;
                        techArr[TI.CVT ] = false;
                        techArr[TI.DCT6] = false;
                    }
                    else if (i == TI.CVT)
                    {
                        techArr[TI.AT6 ] = false;
                        techArr[TI.AT6P] = false;
                        techArr[TI.AT8 ] = false;
                        techArr[TI.AT8P] = false;
                        techArr[TI.DCT6] = false;
                        techArr[TI.DCT8] = false;
                    }
                    else if (i == TI.AT6 || i == TI.AT6P || i == TI.AT8 || i == TI.AT8P)
                    {
                        techArr[TI.DCT6] = false;
                        techArr[TI.DCT8] = false;
                        techArr[TI.AT6 ] = false;
                        //
                        if (i == TI.AT8 || i == TI.AT8P)
                        {
                            techArr[TI.AT6P] = false;
                            techArr[TI.AT8 ] = false;
                            techArr[TI.CVT ] = false;
                        }
                    }
                }
            }
            //
            // --------------------------------------------------
            // rectify electrification technology dependencies
            // --------------------------------------------------
            //
            else if (i == TI.BISG || i == TI.CISG)
            {
                if (techUsed[i])
                {   // BISG and CISG are mutually exclusive --
                    // if either is used, disable entire electrification path, since BISG/CISG are end-of-line
                    RectifyDependencies_DisableTechPaths(techAvailable, techArr, false, false, false, false, false, false, true, false, false, false, false, false, false);
                }
            }
            //
            // --------------------------------------------------
            // rectify strong hybrid technology dependencies
            // --------------------------------------------------
            //
            else if (TI.IsHybridElectricPath(i) || TI.IsAdvancedHybridElecPath(i))
            {
                //// rectify SHEV dependencies (decides whether to enable PHEVs and later if SHEVs were satisfied)
                //if ((i == TI.SHEVP2 || i == TI.SHEVPS) && updateTechEnabled && !afterShevRectified)
                //{
                //    RectifyDependencies_EnableAfterSHEV(techArr, techUsed, vehicle, year, techIdx, settings);
                //    afterShevRectified = true; // to prevent redundant checks when examining multiple techs
                //}
                if (techUsed[i])
                {
                    bool isSHEVP2 = (i == TI.SHEVP2);
                    // basic engine path, manual/auto transmission paths, and electricfication paths
                    // cannot be applied once any of hev become used (except for SHEVP2, which does
                    // not preclude basic-eng, turbo-eng, or adv-eng paths)
                    //  -AND-
                    // turbo-eng, adv-eng, dsl-eng, and shev are mutually exclusive (except for SHEVP2,
                    // which does not preclude basic-eng, turbo-eng, or adv-eng paths)
                    //  -AND-
                    // SHEVP2 and SHEVPS are mutually exclusive
                    //
                    // for simplicity, both P2 and PS are disabled since IsHybridElectricPath(i) or
                    // IsAdvancedHybridElecPath(i) checks ensure SHEV or greater is being evaluated
                    //  - if i is P2 or PS -- OK to disable both due to exclusivity and since the model
                    //    disables applied tech anyway
                    //  - if i is >PHEV -- ok to disable both due to sequential application of hev path
                    RectifyDependencies_DisableTechPaths(techAvailable, techArr, !isSHEVP2, !isSHEVP2, !isSHEVP2, true, true, true, true, true, false, false, false, false, false);

                    // also enforce consistency after SHEV by disabling preceding technologies
                    if (i == TI.PHEV50 || i == TI.BEV200 || i == TI.FCV)
                    {
                        // - no need to check if PHEV30 is being examined -- SHEV check above and PHEV50+
                        //   checks disable it as required
                        // - if PHEV50 is examined, disable both PHEVs
                        // - if EV or FCV is used, disable everything
                        techArr[TI.PHEV30] = false;
                        techArr[TI.PHEV50] = false;
                        if (i == TI.BEV200 || i == TI.FCV)
                        {   // EV200 and FCV are mutually exclusive and are end-of-line technologies
                            // if one or the other is used, disable both
                            techArr[TI.BEV200] = false;
                            techArr[TI.FCV   ] = false;
                        }
                    }
                }
            }
            //
            // --------------------------------------------------
            // rectify other (vehicle) technology dependencies
            // --------------------------------------------------
            //
            //// --- dynamic load reduction techs ---
            //else if (TI.IsDLRPath(i))
            //{   // no dependency checks for dynamic load reduction techs
            //}
            //// --- low rolling resistance tires techs ---
            //else if (TI.IsROLLPath(i))
            //{   // no dependency checks for low rolling resistance tires techs
            //}
            //// --- mass reduction techs ---
            //else if (TI.IsMassReductionPath(i))
            //{   // no dependency checks for mass reduction techs
            //}
            //// --- aerodynamic improvements techs ---
            //else if (TI.IsAEROPath(i))
            //{   // no dependency checks for aerodynamic improvements techs
            //}
        }
        /// <summary>
        /// Disables technologies on specified paths.
        /// The 'techAE' array contains tech-available or tech-enabled states (depending on whether a scenario is being initialized
        /// or a specific model year is being evaluated).
        /// Additionally, a duplicate 'techAvailable' array is accepted and is also updated to ensure that incompatible technologies
        /// are permanently disabled (such as those that are no longer compatible with the vehicle's configuration) for the entire
        /// scenario (all model years) and are not subsequently reset by multiyear. However, the 'techAvailable' array should be
        /// specified as null if the technology paths are being disabled temporarily (such as the criteria for enabling of the paths
        /// is not yet satisfied).
        /// </summary>
        static void RectifyDependencies_DisableTechPaths(
            bool[] techAvailable,
            bool[] techAE,
            bool engBasic, bool engTurbo, bool engAdv, bool engDsl,
            bool trnManual, bool trnAuto,
            bool vehElec, bool vehHEV, bool vehAdvHEV,
            bool vehDLR, bool vehROLL, bool pltAero, bool pltMR)
        {
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                if ((engBasic  && TI.IsBasicEnginePath       (i)) ||
                    (engTurbo  && TI.IsTurboEnginePath       (i)) ||
                    (engAdv    && TI.IsAdvancedEnginePath    (i)) ||
                    (engDsl    && TI.IsDieselEnginePath      (i)) ||
                    (trnManual && TI.IsManualTransmissionPath(i)) ||
                    (trnAuto   && TI.IsAutoTransmissionPath  (i)) ||
                    (vehElec   && TI.IsElectrificationPath   (i)) ||
                    (vehHEV    && TI.IsHybridElectricPath    (i)) ||
                    (vehAdvHEV && TI.IsAdvancedHybridElecPath(i)) ||
                    (vehDLR    && TI.IsDLRPath               (i)) ||
                    (vehROLL   && TI.IsROLLPath              (i)) ||
                    (pltAero   && TI.IsAEROPath              (i)) ||
                    (pltMR     && TI.IsMassReductionPath     (i)))
                {
                    // disable available (if initializing) or enabled (if analyzing model year) state of
                    // incompatible technologies of incompatible technologies
                    techAE[i] = false;
                    if (techAvailable != null)
                    {   // also force disable general availability of incompatible technologies (so that
                        // multiyear cannot apply blocked technologies)
                        techAvailable[i] = false;
                    }
                }
            }
        }
        static void RectifyDependencies_EnableAfterBasicEng(bool[] techEnabled, bool[] techUsed, Vehicle veh, ModelYear year,
            int techIdx, ModelingSettings settings)
        {
            // once the basic engine path is complete, turbo-eng, adv-eng, dsl-eng, and shev paths may be unlocked
            // however, these paths are mutually exclusive, so if a tech in one path is used, others will be disabled
            bool[] techAvailable = veh.ModelData.TechAvailable;
            bool basicEngComplete =
                (techUsed[TI.VVT ] || ! techAvailable[TI.VVT ]) &&
                (techUsed[TI.VVL ] || ! techAvailable[TI.VVL ]) &&
                (techUsed[TI.SGDI] || ! techAvailable[TI.SGDI]) &&
                (techUsed[TI.DEAC] ||
                 techUsed[TI.HCR ] || !(techAvailable[TI.DEAC] || techAvailable[TI.HCR]));
            //
            if (!basicEngComplete)
            {   // the basic eng path is not complete -- disable dependent paths
                RectifyDependencies_DisableTechPaths(null, techEnabled, false, false, false, true, false, false, false, false, true, false, false, false, false);
                // disable SHEVPS (cannot disable entire HEV path since SHEVP2 can be applied)
                techEnabled[TI.SHEVPS] = false;
            }
            else if (techIdx != -1)
            {   // basic eng path is complete -- try enabling dependent paths
                //  - if 'techIdx' = -1, the model is examining all techs from "Enable" method -- in which case, required
                //    techs would be auto-enabled and this call is redundant
                //  - if 'techIdx' <> -1, the model just applied an engine tech, and the "Enable" method is examining
                //    just that one technology -- in which case, required techs would not be auto-enabled
                //
                // do not enable advanced engine, diesel, or hybrid (other HEV tech) paths if SHEVP2 is used
                bool isSHEVP2 = techUsed[TI.SHEVP2];
                for (int i = 0; i < TI.TechnologyCount; i++)
                {
                    if ((TI.IsDieselEnginePath      (i) && !isSHEVP2) ||
                        (TI.IsHybridElectricPath    (i) && !isSHEVP2) ||
                         TI.IsAdvancedHybridElecPath(i))
                    {
                        TechnologyApplicability.Enable(veh, year, i, settings);
                    }
                }
            }
        }
        //static void RectifyDependencies_EnableAfterSHEV(bool[] techEnabled, bool[] techUsed, Vehicle veh, ModelYear year,
        //    int techIdx, ModelingSettings settings)
        //{
        //    // PHEV30 and later techs on hybrid/electric path cannot be applied until either SHEVP2
        //    // or SHEVPS has been installed on a vehicle -- this is because SHEV variants are mutually
        //    // exclusive with hard-coded logic and restrictions to make conversion to PHEV work
        //    //
        //    // if any of the advanced hybrid/electric technologies are already in use, assume SHEV
        //    // path has been completed
        //    //
        //    // enabling logic varies for PS and P2:
        //    //  - if SHEVPS becomes USED, no further checks required
        //    //  - if SHEVP2 becomes USED, the following conditions must be met as well:
        //    //      1) basic engine path is also complete
        //    //      2) no technology on the turbo engine path is in use
        //    bool[] techAvailable = veh.ModelData.TechAvailable;
        //    bool shevComplete = false;
        //    if (techUsed[TI.SHEVPS] ||
        //        techUsed[TI.PHEV30] || techUsed[TI.PHEV50] || techUsed[TI.BEV200] || techUsed[TI.FCV])
        //    {
        //        shevComplete = true;
        //    }
        //    else if (techUsed[TI.SHEVP2])
        //    {
        //        bool basicEngComplete =
        //            (techUsed[TI.VVT ] || ! techAvailable[TI.VVT ]) &&
        //            (techUsed[TI.VVL ] || ! techAvailable[TI.VVL ]) &&
        //            (techUsed[TI.SGDI] || ! techAvailable[TI.SGDI]) &&
        //            (techUsed[TI.DEAC] ||
        //             techUsed[TI.HCR ] || !(techAvailable[TI.DEAC] || techAvailable[TI.HCR]));
        //        //
        //        //bool turboEngUsed = false;
        //        //if (basicEngComplete)
        //        //{
        //        //    for (int i = 0; i < TI.TechnologyCount; i++)
        //        //    {
        //        //        if (TI.IsTurboEnginePath(i) && techUsed[i]) { turboEngUsed = true; break; }
        //        //    }
        //        //}
        //        //
        //        shevComplete = basicEngComplete;// && !turboEngUsed;
        //    }
        //    //
        //    if (!shevComplete)
        //    {   // shev path is not complete -- disable advanced HEV path
        //        RectifyDependencies_DisableTechPaths(null, techEnabled, false, false, false, false, false, false, false, false, true, false, false, false, false);
        //    }
        //    else if (techIdx != -1)
        //    {   // shev path complete -- try enabling advanced HEV path
        //        for (int i = 0; i < TI.TechnologyCount; i++)
        //        {
        //            if (TI.IsAdvancedHybridElecPath(i))
        //            {
        //                TechnologyApplicability.Enable(veh, year, i, settings);
        //            }
        //        }
        //    }
        //}

        #endregion

        #region /* Technology "Weight" Calculations */

        static double GetDeltaWeight(TechnologyInfo tech, Vehicle veh, ModelYear year)
        {
            var    attr    = tech.GetAttributes(veh);
            double gliderW = veh.Description.GliderWeightMR0;
            double deltaCW = ((attr.DeltaWeightPCT != 0) ? attr.DeltaWeightPCT * gliderW :
                              (attr.DeltaWeightLBS != 0) ? attr.DeltaWeightLBS : 0);
            return deltaCW;
        }
        /// <summary>
        /// Determines the vehicle's new weight metrics (curb weight, GVWR, and GCWR) as a result of applying the specified
        /// technology to the given vehicle.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the new weight metrics.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the new weight metrics.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="impact">The accumulated impact of technology application on a vehicle.</param>
        /// <param name="deltaCW">When this function returns, specifies what the change in vehicle's curb weight will be as a
        ///   result of applying the specified technology.</param>
        /// <param name="deltaGVWR">When this function returns, specifies what the change in vehicle's GVWR will be as a result
        ///   of applying the specified technology.</param>
        /// <param name="deltaGCWR">When this function returns, specifies what the change in vehicle's GCWR will be as a result
        ///   of applying the specified technology.</param>
        public static void GetWeightChange(TechnologyInfo tech, Vehicle veh, Scenario scen, ModelYear year, TechImpact impact,
            out double deltaCW, out double deltaGVWR, out double deltaGCWR)
        {
            var vd = veh.Description;

            // compute delta CW and set delta GVWR/GCWR to 0 initially
            deltaCW   = GetDeltaWeight(tech, veh, year);
            deltaGVWR = 0;
            deltaGCWR = 0;

            // calc current weight info, accounting for any pending changes
            double vehCW   = vd.CurbWeight + impact.DeltaCurbWeight[year.Index];
            double vehGVWR = vd.GVWR       + impact.DeltaGVWR      [year.Index];
            double vehGCWR = vd.GCWR       + impact.DeltaGCWR      [year.Index];

            // update new variables if there is a change in CW
            if (deltaCW != 0)
            {
                if (TI.IsMassReductionPath(tech.Index) && veh.RegClass == RegulatoryClass.LightTruck2b3)
                {
                    double p = 1 - scen.ScenInfo[veh.RegClass].PayloadReturn[year.Index];
                    double t = 1 - scen.ScenInfo[veh.RegClass].TowingReturn [year.Index];
                    double gvwrLBound = (veh.RegClass == RegulatoryClass.LightTruck2b3) ? 8501 : 0;
                    //
                    deltaGVWR = vehGVWR - Math.Max(gvwrLBound,
                                          Math.Min(vehGVWR - p * deltaCW  , (vehCW   - deltaCW  ) * vd.MaxGVWRToCW  ));
                    deltaGCWR = vehGCWR - Math.Min(vehGCWR - t * deltaGVWR, (vehGVWR - deltaGVWR) * vd.MaxGCWRToGVWR);
                }
                else if (TI.IsMassReductionPath(tech.Index) && veh.TechClassString == "Pickup")
                {
                    deltaGVWR = deltaCW;
                }
            }
        }

        #endregion

        #region /* Technology Improvement/Cost Calculations */

        #region /* GetImprovement methods */

        /// <summary>
        /// Computes the fuel consumption estimate of the specified technology for the given vehicle.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the fuel consumption improvement.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the fuel consumption improvement.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used for modeling.</param>
        /// <param name="impact">The accumulated impact of technology application on a vehicle.</param>
        /// <param name="isSHEVP2Selected">true, if SHEVP2 has been selected for application via backfill; false, otherwise.</param>
        /// <param name="fc1">When this function returns, contains the fuel consumption estimate of the specified technology.</param>
        /// <param name="fc2">When this function returns, contains the fuel consumption estimate of the specified technology when
        ///   operating on secondary fuel (a value is returned if applicable; 0 if not applicable).</param>
        /// <param name="fs2">When this function returns, contains the fuel share of the specified technology when operating on
        ///   secondary fuel (a value is returned if applicable; 0 if not applicable).</param>
        /// <param name="fcAdj">When this function returns, contains the fuel consumption adjustment (or correction) factor
        ///   resulting from application of the specified technology.</param>
        /// <param name="fcPwrAdj">When this function returns, contains the fuel consumption adjustment (or correction) factor,
        ///   due to differences between simulated vehicle's horsepower and actual vehicle's horsepower, resulting from application
        ///   of the specified technology.</param>
        /// <param name="deltaPwr">When this function returns, contains the change vehicle power resulting from application of the
        ///   specified technology.</param>
        public static void GetImprovement(TechnologyInfo tech, Vehicle veh, Scenario scen, ModelYear year, ModelingSettings settings,
            TechImpact impact, bool isSHEVP2Selected,
            out double fc1, out double fc2, out double fs2, out double fcAdj, out double fcPwrAdj, out double deltaPwr)
        {
            Technologies  techData = settings.Technologies;
            OperatingModes opModes = settings.OperatingModes;

            // compute estimated improvement
            TechnologyAttributes ta = tech.GetAttributes(veh);
            GetEstimatedFC(tech, veh, scen, year, techData, isSHEVP2Selected, out fc1, out fc2);
            fs2 = ta.FSSecondary;

            // detect gaps for basic engine techs (VVT, VVL, SGDI, and DEAC) -- these may occur out of order on an engine
            // skip FC adj factors for those techs, since there is no match in the DB and the FC-adj-key cannot be computed
            bool skipFCAdj = false;
            if (opModes.BackfillMissingBEP)
            {
                bool[] techUsed = veh.ModelData.TechUsed;
                //
                if      (tech.Index == TI.VVT ) { skipFCAdj = techUsed[TI.DEAC] || techUsed[TI.SGDI] || techUsed[TI.VVL]; }
                else if (tech.Index == TI.VVL ) { skipFCAdj = techUsed[TI.DEAC] || techUsed[TI.SGDI]; }
                else if (tech.Index == TI.SGDI) { skipFCAdj = techUsed[TI.DEAC]; }
            }

            // compute FC adjutment factor
            if (!skipFCAdj && TI.IsFCAdjKey(tech.Index))
            {   // tech being evaluated uses FC adjustments --
                ulong key0 = veh.ModelData.FCAdjKey;
                ulong key1 = GetFCAdjKey(veh, tech);
                // get current and new factors and obtain ratio
                double fcAdj0 = techData.GetFCAdjFactors(key0, veh.TechnologyClass);
                double fcAdj1 = techData.GetFCAdjFactors(key1, veh.TechnologyClass);
                fcAdj = fcAdj1 / fcAdj0;
            }
            else
            {   // tech being evaluated is a free floating technology w/o adj factors -- set adj ratios to 1
                fcAdj = 1;
            }

            // power adjustments haven't been implemented -- these values exists as place holders
            fcPwrAdj = 1;
            deltaPwr = 0;
        }
        static void GetEstimatedFC(TechnologyInfo tech, Vehicle veh, Scenario scen, ModelYear year, Technologies techData,
            bool isSHEVP2Selected, out double fc1, out double fc2)
        {
            // get primary and secondary FC
            TechnologyAttributes attr = tech.GetAttributes(veh);
            fc1 = 1 - attr.FC;
            fc2 = 1 - attr.FCSecondary;

            // consider special cases for some technologies
            int    techClass      = veh.TechnologyClass;
            bool[] techUsed       = veh.ModelData.TechUsed;
            bool[] techApplied    = veh.ModelData.TechApplied;
            bool[] techSuperseded = veh.ModelData.TechSuperseded;
            //if (TI.IsMassReductionPath(tech.Index))
            //{   // adjust FC based on vehicle's weight, using % of base veh's curb weight, absolute weight change, or none
            //    double newCW, newGVWR, newGCWR;
            //    GetWeightChange(tech, veh, scen, year, out newCW, out newGVWR, out newGCWR);

            //    var    vd      = veh.Description;
            //    double deltaCW = (vd.CurbWeight - newCW) / vd.BaseWeight;
            //    if (deltaCW != 0)
            //    {
            //        double baseTW  = Standards.GetTestWeight(veh, scen, year, vd.BaseWeight, vd.BaseGVWR, vd.BaseGCWR, 0);
            //        double vehTW   = Standards.GetTestWeight(veh, scen, year, vd.CurbWeight, vd.GVWR    , vd.GCWR    , 0);
            //        double newTW   = Standards.GetTestWeight(veh, scen, year, newCW        , newGVWR    , newGCWR    , 0);
            //        double deltaTW = (vehTW - newTW) / baseTW;
            //        if (fc1 != 0) { fc1 = deltaTW * (fc1 / deltaCW); }
            //        if (fc2 != 0) { fc2 = deltaTW * (fc2 / deltaCW); }
            //    }
            //}
            if (tech.Index == TI.ADSL || tech.Index == TI.TURBO1)
            {
                //----------------------------------------------------------------------------------//
                // Conversion to ADSL or TURBO1 requires special handling.                          //
                //                                                                                  //
                // The FC incremental values for ADSL and TURBO1 are specified over DEAC. Thus, if  //
                // a vehicle is using DEAC, no additional logic is required. However, if a vehicle  //
                // is using HCR and/or HCRP, the model should back out all FC data for either used  //
                // HCR tech and add in FC data for the DEAC tech.                                   //
                //----------------------------------------------------------------------------------//
                if ( techUsed[TI.HCR ]) { fc1 = fc1 / (1 - techData.GetFC(TI.HCR , veh)); }
                if ( techUsed[TI.HCRP]) { fc1 = fc1 / (1 - techData.GetFC(TI.HCRP, veh)); }
                if (!techUsed[TI.DEAC]) { fc1 = fc1 * (1 - techData.GetFC(TI.DEAC, veh)); }
            }
            else if (tech.Index == TI.SHEVP2 || tech.Index == TI.SHEVPS)
            {
                //----------------------------------------------------------------------------------//
                // Conversion to SHEV requires special handling.                                    //
                //                                                                                  //
                // When converting to SHEV, the model should back out FC data for all automatic     //
                // transmissions, except for the lowest level (currently AT5). The FC values will   //
                // be removed only for those technologies that are on the "USED path" for a vehicle.//
                // Additionally, for SHEVP2, FC value for DCT6 should be added back, since SHEVP2   //
                // is simulated as using that transmission.                                         //
                //                                                                                  //
                // Furthermore, SHEVPS is defined as incremental over DEAC. If a vehicle is using   //
                // HCR, HCRP, and/or HCR2, the model should back out all FC data for any used HCR   //
                // tech and add in FC data for the DEAC tech. The model should also back out any    //
                // TURBO path technology present on the vehicle before converting to SHEVPS.        //
                //                                                                                  //
                // Lastly, SHEVP2 is defined as incremental over BISG, while SHEVPS is defined as   //
                // incremental over CISG. Perform additional adjustments to ensure consistency for  //
                // incremental FC improvements when converting to either SHEV tech.                 //
                //----------------------------------------------------------------------------------//
                if      (techUsed[TI.DCT8]) { fc1 = fc1 / (1 - techData.GetFC(TI.DCT6, veh)) / (1 - techData.GetFC(TI.DCT8, veh)); }
                else if (techUsed[TI.DCT6]) { fc1 = fc1 / (1 - techData.GetFC(TI.DCT6, veh)); }
                else if (techUsed[TI.CVT ]) { fc1 = fc1 / (1 - techData.GetFC(TI.AT6 , veh)) / (1 - techData.GetFC(TI.CVT , veh)); }
                else if (techUsed[TI.AT8 ]) { fc1 = fc1 / (1 - techData.GetFC(TI.AT6 , veh)) / (1 - techData.GetFC(TI.AT8 , veh)); }
                else if (techUsed[TI.AT6 ]) { fc1 = fc1 / (1 - techData.GetFC(TI.AT6 , veh)); }
                // also back out AT6P and AT8P, only if they were applied by the model
                if (techApplied[TI.AT6P]) { fc1 = fc1 / (1 - techData.GetFC(TI.AT6P, veh)); }
                if (techApplied[TI.AT8P]) { fc1 = fc1 / (1 - techData.GetFC(TI.AT8P, veh)); }
                //
                if (tech.Index == TI.SHEVP2)
                {
                    if (techUsed[TI.CISG])
                    {   // remove CISG and add BISG FC improvement
                        fc1 = fc1 / (1 - techData.GetFC(TI.CISG, veh)) * (1 - techData.GetFC(TI.BISG, veh));
                    }
                    // add back DCT6 FC improvement
                    fc1 *= (1 - techData.GetFC(TI.DCT6, veh));
                }
                else if (tech.Index == TI.SHEVPS)
                {
                    if (techUsed[TI.BISG])
                    {   // remove BISG and add CISG FC improvement
                        fc1 = fc1 / (1 - techData.GetFC(TI.BISG, veh)) * (1 - techData.GetFC(TI.CISG, veh));
                    }
                    if (techUsed[TI.HCR2])
                    {   // remove HCR2 if it is used
                        fc1 = fc1 / (1 - techData.GetFC(TI.HCR2, veh));
                    }
                    // check TURBO engine path and remove the latest technology marked as USED along with any
                    // preceding ones that are listed as applicable
                    bool turboRemoved = GetEstimatedFC_RemoveTurbo(veh, techData, techUsed, ref fc1, ref fc2);
                    if (!turboRemoved && (techUsed[TI.HCR] || techUsed[TI.HCRP]))
                    {   // remove HCR/HCRP and add DEAC
                        // if vehicle has anything on the TURBO engine path, this adjustment isn't necessary,
                        // since conversion to TURBO1 tech would have already accounted for HCR+HCRP/DEAC split
                        if ( techUsed[TI.HCR ]) { fc1 = fc1 / (1 - techData.GetFC(TI.HCR , veh)); }
                        if ( techUsed[TI.HCRP]) { fc1 = fc1 / (1 - techData.GetFC(TI.HCRP, veh)); }
                        if (!techUsed[TI.DEAC]) { fc1 = fc1 * (1 - techData.GetFC(TI.DEAC, veh)); }
                    }
                }
            }
            else if (tech.Index == TI.PHEV30)
            {
                //----------------------------------------------------------------------------------//
                // Conversion to PHEV30 requires special handling.                                  //
                //                                                                                  //
                // The FC incremental value for PHEV30 is specified over SHEVPS. Thus, if a vehicle //
                // is coming from SHEVPS, no additional logic is required. However, if a vehicle is //
                // coming from SHEVP2, the model should back out all FC data for the DCT6 technology//
                // and the BISG/SHEVP2 path, and add in FC data for the CISG/SHEVPS path. If SHEVP2 //
                // uses HCR engine technology, the model should also back out FC data for any used  //
                // HCR technology and add in FC data for the DEAC technology. Lastly, the model     //
                // should also back out any TURBO path technology present on the vehicle before     //
                // converting to PHEV30.                                                            //
                //----------------------------------------------------------------------------------//
                if (techUsed[TI.SHEVP2] || isSHEVP2Selected)
                {
                    double tFC;
                    //
                    tFC = (1 - techData.GetFC(TI.DCT6  , veh)); fc1 /= tFC; fc2 /= tFC;
                    tFC = (1 - techData.GetFC(TI.BISG  , veh)); fc1 /= tFC; fc2 /= tFC;
                    tFC = (1 - techData.GetFC(TI.SHEVP2, veh)); fc1 /= tFC; fc2 /= tFC;
                    tFC = (1 - techData.GetFC(TI.CISG  , veh)); fc1 *= tFC; fc2 *= tFC;
                    tFC = (1 - techData.GetFC(TI.SHEVPS, veh)); fc1 *= tFC; fc2 *= tFC;
                    if (techUsed[TI.HCR2])
                    {   // remove HCR2 if it is used
                        tFC = (1 - techData.GetFC(TI.HCR2, veh)); fc1 /= tFC; fc2 /= tFC;
                    }
                    // check TURBO engine path and remove the latest technology marked as USED along with any
                    // preceding ones that are listed as applicable
                    bool turboRemoved = GetEstimatedFC_RemoveTurbo(veh, techData, techUsed, ref fc1, ref fc2);
                    if (!turboRemoved && (techUsed[TI.HCR] || techUsed[TI.HCRP]))
                    {   // remove HCR and add DEAC
                        // if vehicle has anything on the TURBO engine path, this adjustment isn't necessary,
                        // since conversion to TURBO1 tech would have already accounted for HCR+HCRP/DEAC split
                        if ( techUsed[TI.HCR ]) { tFC = (1 - techData.GetFC(TI.HCR , veh)); fc1 /= tFC; fc2 /= tFC; }
                        if ( techUsed[TI.HCRP]) { tFC = (1 - techData.GetFC(TI.HCRP, veh)); fc1 /= tFC; fc2 /= tFC; }
                        if (!techUsed[TI.DEAC]) { tFC = (1 - techData.GetFC(TI.DEAC, veh)); fc1 *= tFC; fc2 *= tFC; }
                    }
                }
            }

            // if the technology being applied supersedes any FCTimeBased technologies, negate FC improvement
            // of the superseded techs
            int[] supersededTechs = TechSupersedes[tech.Index];
            for (int i = 0; i < supersededTechs.Length; i++)
            {
                int stIndex = supersededTechs[i];
                if (  TI.IsFCTimeBased      (stIndex) &&
                      techUsed              [stIndex] &&
                     !techSuperseded        [stIndex] &&
                    (!TI.IsEngineLevel      (stIndex) || veh.Engine       != null) &&
                    (!TI.IsTransmissionLevel(stIndex) || veh.Transmission != null) &&
                    (!TI.IsPlatformLevel    (stIndex) || veh.Platform     != null))
                {
                    double lastFC, currentFC;
                    GetEstimatedFCTimeBased(year, veh, techData.TechnologyList[stIndex], out lastFC, out currentFC);
                    fc1 /= (1 - currentFC);
                    fc2 /= (1 - currentFC);
                }
            }

            // finalize FC values
            fc1 = 1 - fc1;
            fc2 = 1 - fc2;
        }
        static bool GetEstimatedFC_RemoveTurbo(Vehicle veh, Technologies techData, bool[] techUsed, ref double fc1, ref double fc2)
        {
            double fcTmp;
            if (techUsed[TI.CEGR2])
            {
                fcTmp = 1 / (1 - techData.GetFC(TI.CEGR2 , veh))
                          / (1 - techData.GetFC(TI.CEGR1P, veh))
                          / (1 - techData.GetFC(TI.CEGR1 , veh))
                          / (1 - techData.GetFC(TI.TURBO2, veh))
                          / (1 - techData.GetFC(TI.SEGR  , veh))
                          / (1 - techData.GetFC(TI.DWSP  , veh))
                          / (1 - techData.GetFC(TI.TURBO1, veh));
                fc1 *= fcTmp;
                fc2 *= fcTmp;
                return true;
            }
            else if (techUsed[TI.CEGR1P])
            {
                fcTmp = 1 / (1 - techData.GetFC(TI.CEGR1P, veh))
                          / (1 - techData.GetFC(TI.CEGR1 , veh))
                          / (1 - techData.GetFC(TI.TURBO2, veh))
                          / (1 - techData.GetFC(TI.SEGR  , veh))
                          / (1 - techData.GetFC(TI.DWSP  , veh))
                          / (1 - techData.GetFC(TI.TURBO1, veh));
                fc1 *= fcTmp;
                fc2 *= fcTmp;
                return true;
            }
            else if (techUsed[TI.CEGR1])
            {
                fcTmp = 1 / (1 - techData.GetFC(TI.CEGR1 , veh))
                          / (1 - techData.GetFC(TI.TURBO2, veh))
                          / (1 - techData.GetFC(TI.SEGR  , veh))
                          / (1 - techData.GetFC(TI.DWSP  , veh))
                          / (1 - techData.GetFC(TI.TURBO1, veh));
                fc1 *= fcTmp;
                fc2 *= fcTmp;
                return true;
            }
            else if (techUsed[TI.TURBO2])
            {
                fcTmp = 1 / (1 - techData.GetFC(TI.TURBO2, veh))
                          / (1 - techData.GetFC(TI.SEGR  , veh))
                          / (1 - techData.GetFC(TI.DWSP  , veh))
                          / (1 - techData.GetFC(TI.TURBO1, veh));
                fc1 *= fcTmp;
                fc2 *= fcTmp;
                return true;
            }
            else if (techUsed[TI.SEGR])
            {
                fcTmp = 1 / (1 - techData.GetFC(TI.SEGR  , veh))
                          / (1 - techData.GetFC(TI.DWSP  , veh))
                          / (1 - techData.GetFC(TI.TURBO1, veh));
                fc1 *= fcTmp;
                fc2 *= fcTmp;
                return true;
            }
            else if (techUsed[TI.DWSP])
            {
                fcTmp = 1 / (1 - techData.GetFC(TI.DWSP  , veh))
                          / (1 - techData.GetFC(TI.TURBO1, veh));
                fc1 *= fcTmp;
                fc2 *= fcTmp;
                return true;
            }
            else if (techUsed[TI.TURBO1])
            {
                fcTmp = 1 / (1 - techData.GetFC(TI.TURBO1, veh));
                fc1 *= fcTmp;
                fc2 *= fcTmp;
                return true;
            }
            return false;
        }
        internal static void GetEstimatedFCTimeBased(ModelYear year, Vehicle veh, TechnologyInfo tech, out double lastFC, out double currentFC)
        {
            Vehicle ldr;
            int ldrLastMY, ldrCurrMY;
            GetTimeBasedTechYearsRange(year, veh, tech, out ldr, out ldrLastMY, out ldrCurrMY);

            // obtain FC values at leader's current and last model years of technology availability
            TechnologyAttributes attr = tech.GetAttributes(ldr);
            int    baseYear = attr.YearAvailable;
            double step     = 1;
            double fc       = attr.FC;
            //
            lastFC          = Math.Ceiling(Math.Max(1 + ldrLastMY - baseYear, 0) / step) * fc;
            currentFC       = Math.Ceiling(Math.Max(1 + ldrCurrMY - baseYear, 0) / step) * fc;
        }

        #endregion

        #region /* GetCost methods */

        /// <summary>
        /// Returns the cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the cost.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the cost of the <see cref="TechnologyInfo"/>.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="techData">An object containing technology attributes, costs, and adjustment factors.</param>
        /// <param name="startYear">The value of the model year when modeling began.</param>
        /// <param name="endYear">The value of the model year when modeling ends.</param>
        /// <returns>The cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.</returns>
        public static double GetCost(TechnologyInfo tech, Vehicle veh, ModelYear year, Technologies techData, int startYear, int endYear)
        {
            double erpCost;
            bool[] committedERPCosts = null;
            return GetCost(tech, veh, year, techData, false, 0, startYear, endYear, false, out erpCost, ref committedERPCosts);
        }
        /// <summary>
        /// Returns the cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the cost.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the cost of the <see cref="TechnologyInfo"/>.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="techData">An object containing technology attributes, costs, and adjustment factors.</param>
        /// <param name="isSHEVP2Selected">true, if SHEVP2 has been selected for application via backfill; false, otherwise.</param>
        /// <param name="yearOfInitialApplication">If the technology is applied due to multiyear, specifies the initial year when
        ///   the technology was installed on a vehicle.</param>
        /// <param name="startYear">The value of the model year when modeling began.</param>
        /// <param name="endYear">The value of the model year when modeling ends.</param>
        /// <param name="strandedCapital">When this function returns, contains the stranded capital costs accrued as a result of
        ///   applying this technology.</param>
        /// <param name="committedStrandedCapital">An array indicating whether the stranded capital cost of a technology at a
        ///   specific index was already considered by an earlier technology found in techGroup (for example, if one of the
        ///   backfilled technologies included the stranded capital for replacing a technology that it superseded, we do not want
        ///   to count the stranded capital again when evaluating the cost of the current technology).</param>
        /// <returns>The cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.</returns>
        public static double GetCost(TechnologyInfo tech, Vehicle veh, ModelYear year, Technologies techData, bool isSHEVP2Selected,
            int yearOfInitialApplication, int startYear, int endYear, out double strandedCapital, ref bool[] committedStrandedCapital)
        {
            return GetCost(tech, veh, year, techData, isSHEVP2Selected, yearOfInitialApplication, startYear, endYear, true,
                out strandedCapital, ref committedStrandedCapital);
        }
        static double GetCost(TechnologyInfo tech, Vehicle veh, ModelYear year, Technologies techData, bool isSHEVP2Selected,
            int yearOfInitialApplication, int startYear, int endYear, bool isCF, out double strandedCapital, ref bool[] committedStrandedCapital)
        {
            //----------------------------------------------------------------------------------//
            // Clarification of certain parameters:                                             //
            //  isCF:  if this value is true, GetCost is called when evaluating CFs; if this    //
            //         value is false, GetCost is called during model year pre-processing in    //
            //         order to adjust for learning                                             //
            //----------------------------------------------------------------------------------//
            //
            // compute estimated costs, learning rate, and synergy effect
            double techCost = GetEstimatedCost(tech, veh, year, techData, isSHEVP2Selected);
            double synergy  = GetCostSynergy  (tech, veh, isCF, techData);
            strandedCapital = GetStrandedCapitalCost(tech, veh, year, techData, yearOfInitialApplication, isCF, ref committedStrandedCapital);
            // return combined effect of base cost, synergy, and stranded capital
            return techCost + synergy + strandedCapital;
        }
        static double GetEstimatedCost(TechnologyInfo tech, Vehicle veh, ModelYear year, Technologies techData, bool isSHEVP2Selected)
        {
            // calculate base technology cost
            double cost = GetEstimatedCost(tech, veh, year);

            // calculate cost adjustments for branch points
            bool[] techUsed       = veh.ModelData.TechUsed;
            bool[] techApplied    = veh.ModelData.TechApplied;
            bool[] techSuperseded = veh.ModelData.TechSuperseded;
            if (tech.Index == TI.ADSL || tech.Index == TI.TURBO1)
            {
                //----------------------------------------------------------------------------------//
                // Conversion to ADSL or TURBO1 requires special handling.                          //
                // (See note in GetEstimatedFC(...) with explanation of logic.)                     //
                //----------------------------------------------------------------------------------//
                if ( techUsed[TI.HCR ]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.HCR ], veh, year); }
                if ( techUsed[TI.HCRP]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.HCRP], veh, year); }
                if (!techUsed[TI.DEAC]) { cost += GetEstimatedCost(techData.TechnologyList[TI.DEAC], veh, year); }
            }
            else if (tech.Index == TI.SHEVP2 || tech.Index == TI.SHEVPS)
            {
                //----------------------------------------------------------------------------------//
                // Conversion to SHEV requires special handling.                                    //
                // (See note in GetEstimatedFC(...) with explanation of logic.)                     //
                //----------------------------------------------------------------------------------//
                if (techUsed[TI.DCT8])
                {
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.DCT8], veh, year);
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.DCT6], veh, year);
                }
                else if (techUsed[TI.DCT6])
                {
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.DCT6], veh, year);
                }
                else if (techUsed[TI.CVT])
                {
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.CVT], veh, year);
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.AT6], veh, year);
                }
                else if (techUsed[TI.AT8])
                {
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.AT8], veh, year);
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.AT6], veh, year);
                }
                else if (techUsed[TI.AT6])
                {
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.AT6], veh, year);
                }
                // also back out AT6P and AT8P, only if they were applied by the model
                if (techApplied[TI.AT6P]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.AT6P], veh, year); }
                if (techApplied[TI.AT8P]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.AT8P], veh, year); }
                //
                if (tech.Index == TI.SHEVP2)
                {
                    if (techUsed[TI.CISG])
                    {   // remove CISG and add BISG
                        cost -= GetEstimatedCost(techData.TechnologyList[TI.CISG], veh, year);
                        cost += GetEstimatedCost(techData.TechnologyList[TI.BISG], veh, year);
                    }
                    // add back DCT6 cost
                    cost += GetEstimatedCost(techData.TechnologyList[TI.DCT6], veh, year);
                }
                else if (tech.Index == TI.SHEVPS)
                {
                    if (techUsed[TI.BISG])
                    {   // remove BISG and add CISG
                        cost -= GetEstimatedCost(techData.TechnologyList[TI.BISG], veh, year);
                        cost += GetEstimatedCost(techData.TechnologyList[TI.CISG], veh, year);
                    }
                    if (techUsed[TI.HCR2])
                    {   // remove HCR2 if it is used
                        cost -= GetEstimatedCost(techData.TechnologyList[TI.HCR2], veh, year);
                    }
                    // check TURBO engine path and remove the latest technology marked as USED along with any
                    // preceding ones that are listed as applicable
                    bool turboRemoved = GetEstimatedCost_RemoveTurbo(veh, year, techData, techUsed, ref cost);
                    if (!turboRemoved && (techUsed[TI.HCR] || techUsed[TI.HCRP]))
                    {   // remove HCR and add DEAC
                        if ( techUsed[TI.HCR ]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.HCR ], veh, year); }
                        if ( techUsed[TI.HCRP]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.HCRP], veh, year); }
                        if (!techUsed[TI.DEAC]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.DEAC], veh, year); }
                    }
                }
            }
            else if (tech.Index == TI.PHEV30)
            {
                //----------------------------------------------------------------------------------//
                // Conversion to PHEV30 requires special handling.                                  //
                // (See note in GetEstimatedFC(...) with explanation of logic.)                     //
                //----------------------------------------------------------------------------------//
                if (techUsed[TI.SHEVP2] || isSHEVP2Selected)
                {
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.DCT6  ], veh, year);
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.BISG  ], veh, year);
                    cost -= GetEstimatedCost(techData.TechnologyList[TI.SHEVP2], veh, year);
                    cost += GetEstimatedCost(techData.TechnologyList[TI.CISG  ], veh, year);
                    cost += GetEstimatedCost(techData.TechnologyList[TI.SHEVPS], veh, year);
                    if (techUsed[TI.HCR2])
                    {   // remove HCR2 if it is used
                        cost -= GetEstimatedCost(techData.TechnologyList[TI.HCR2], veh, year);
                    }
                    // check TURBO engine path and remove the latest technology marked as USED along with any
                    // preceding ones that are listed as applicable
                    bool turboRemoved = GetEstimatedCost_RemoveTurbo(veh, year, techData, techUsed, ref cost);
                    if (!turboRemoved && (techUsed[TI.HCR] || techUsed[TI.HCRP]))
                    {   // remove HCR and add DEAC
                        if ( techUsed[TI.HCR ]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.HCR ], veh, year); }
                        if ( techUsed[TI.HCRP]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.HCRP], veh, year); }
                        if (!techUsed[TI.DEAC]) { cost -= GetEstimatedCost(techData.TechnologyList[TI.DEAC], veh, year); }
                    }
                }
            }

            // if the technology being applied supersedes any FCTimeBased technologies, negate cost value
            // of the superseded techs
            int[] supersededTechs = TechSupersedes[tech.Index];
            for (int i = 0; i < supersededTechs.Length; i++)
            {
                int stIndex = supersededTechs[i];
                if (  TI.IsFCTimeBased      (stIndex) &&
                      techUsed              [stIndex] &&
                     !techSuperseded        [stIndex] &&
                    (!TI.IsEngineLevel      (stIndex) || veh.Engine       != null) &&
                    (!TI.IsTransmissionLevel(stIndex) || veh.Transmission != null) &&
                    (!TI.IsPlatformLevel    (stIndex) || veh.Platform     != null))
                {
                    double lastCost, currentCost;
                    GetEstimatedCostTimeBased(year, veh, techData.TechnologyList[stIndex], out lastCost, out currentCost);
                    cost -= (1 - currentCost);
                }
            }

            // return overall technology cost
            return cost;
        }
        static double GetEstimatedCost(TechnologyInfo tech, Vehicle veh, ModelYear year)
        {
            if (!tech.GetAttributes(veh).Applicable) { return 0; }
            //
            double[] costTable = tech.GetCosts(veh).CostTable;
            int        yrIndex = Math.Min(costTable.Length - 1, year.Index);
            double        cost = costTable[yrIndex];
            //
            // consider special cases for some technologies
            if (TI.IsCostPerPound(tech.Index))
            {   // adjust cost based on vehicle's weight, using % of base glider weight, absolute weight change, or none
                double deltaCW = GetDeltaWeight(tech, veh, year);
                cost *= deltaCW;
            }
            //else if (TI.IsCostPerCylinder(tech.Index))
            //{   // compute cost based on number of engine cylinders
            //    int cylinders = (veh.Engine == null) ? veh.LegacyEng.Cylinders : veh.Engine.Description.Cylinders;
            //    cost *= cylinders;
            //}
            //else if (TI.IsCostPerBank(tech.Index))
            //{   // compute cost based on engine configuration
            //    string engConfig = (veh.Engine == null) ? veh.LegacyEng.Configuration.ToUpper() : veh.Engine.Description.Configuration.ToUpper();
            //    int costMult = (engConfig == "R" || engConfig == "I") ? 1 :
            //        (engConfig == "F" || engConfig == "H" || engConfig == "V") ? 2 : (engConfig == "W") ? 4 : 0;
            //    cost *= costMult;
            //}
            //
            // return the estimated cost
            return cost;
        }
        static bool GetEstimatedCost_RemoveTurbo(Vehicle veh, ModelYear year, Technologies techData, bool[] techUsed, ref double cost)
        {
            if (techUsed[TI.CEGR2])
            {
                cost -= GetEstimatedCost(techData.TechnologyList[TI.CEGR2 ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.CEGR1P], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.CEGR1 ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO2], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.SEGR  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.DWSP  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO1], veh, year);
                return true;
            }
            else if (techUsed[TI.CEGR1P])
            {
                cost -= GetEstimatedCost(techData.TechnologyList[TI.CEGR1P], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.CEGR1 ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO2], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.SEGR  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.DWSP  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO1], veh, year);
                return true;
            }
            else if (techUsed[TI.CEGR1])
            {
                cost -= GetEstimatedCost(techData.TechnologyList[TI.CEGR1 ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO2], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.SEGR  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.DWSP  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO1], veh, year);
                return true;
            }
            else if (techUsed[TI.TURBO2])
            {
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO2], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.SEGR  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.DWSP  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO1], veh, year);
                return true;
            }
            else if (techUsed[TI.SEGR])
            {
                cost -= GetEstimatedCost(techData.TechnologyList[TI.SEGR  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.DWSP  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO1], veh, year);
                return true;
            }
            else if (techUsed[TI.DWSP])
            {
                cost -= GetEstimatedCost(techData.TechnologyList[TI.DWSP  ], veh, year);
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO1], veh, year);
                return true;
            }
            else if (techUsed[TI.TURBO1])
            {
                cost -= GetEstimatedCost(techData.TechnologyList[TI.TURBO1], veh, year);
                return true;
            }
            return false;
        }
        static double GetStrandedCapitalCost(TechnologyInfo tech, Vehicle veh, ModelYear year, Technologies techData,
            int yearOfInitialApplication, bool isCF, ref bool[] committedStrandedCapital)
        {
            double strandedCapital = 0;
            //
            if (isCF)
            {   // evaluating CFs (see note in "GetCost") -- need to examine tech's supersede and soft-supersede lists
                for (int i = 0; i < TechSupersedes2[tech.Index].Length; i++)
                {
                    int sTech = TechSupersedes2[tech.Index][i]; // sTech -- the technology which will be superseded by this tech
                    Vehicle.CModelData vmd = veh.ModelData;
                    bool isCommittedStrandedCapital = (committedStrandedCapital != null && committedStrandedCapital[sTech]);
                    if (!isCommittedStrandedCapital && vmd.TechApplied[sTech] && vmd.TechSupersededYear[sTech] == 0)
                    {   // tech was applied during the modeling process, but has not been superseded yet, and was not committed
                        // to being superseded by an earlier backfilling technology -- get penalty costs
                        TechnologyCosts tc = techData.TechnologyList[sTech].GetCosts(veh);
                        int techAge = yearOfInitialApplication - vmd.TechAppliedYear[sTech] - 1;
                        strandedCapital += (techAge < 0 || techAge >= tc.StrandedCapital.Length) ? 0 : tc.StrandedCapital[techAge];
                        if (committedStrandedCapital != null) { committedStrandedCapital[sTech] = true; }
                    }
                } // next i (technology to supersede)
            }
            else
            {   // adjusting for learning -- no need to examine tech's supersede list
                Vehicle.CModelData vmd = veh.ModelData;
                int supersededYear = vmd.TechSupersededYear[tech.Index];
                if (vmd.TechApplied[tech.Index] && supersededYear > 0)
                {   // tech was applied during the modeling process and was later superseded -- get penalty costs
                    if (year.Year - supersededYear < 5)
                    {
                        TechnologyCosts tc = tech.GetCosts(veh);
                        int techAge = supersededYear - vmd.TechAppliedYear[tech.Index] - 1;
                        strandedCapital = (techAge < 0 || techAge >= tc.StrandedCapital.Length) ? 0 : tc.StrandedCapital[techAge];
                    }
                }
            }
            //
            return strandedCapital;
        }
        internal static void GetEstimatedCostTimeBased(ModelYear year, Vehicle veh, TechnologyInfo tech, out double lastCost, out double currentCost)
        {
            Vehicle ldr;
            int ldrLastMY, ldrCurrMY;
            GetTimeBasedTechYearsRange(year, veh, tech, out ldr, out ldrLastMY, out ldrCurrMY);

            // obtain cost values at leader's current and last model years of technology availability
            int baseYear = tech.GetAttributes(ldr).YearAvailable;
            var costs    = tech.GetCosts     (ldr).CostTable;
            //
            lastCost     = (ldrLastMY < baseYear) ? 0 : costs[ldrLastMY - ModelYear.MinYear];
            currentCost  = (ldrCurrMY < baseYear) ? 0 : costs[ldrCurrMY - ModelYear.MinYear];
        }

        /// <summary>
        /// Calculates the accumulated synergy effect for a technology represetned by the specified techIndex.
        /// </summary>
        static double GetCostSynergy(TechnologyInfo tech, Vehicle veh, bool isCF, Technologies techData)
        {
            double synergy = 0;
            if (isCF)
            {   // technology is being applied during compliance evaluation -- aggregate synergy
                // effect of this technology with all used technologies
                bool[] techUsed = veh.ModelData.TechUsed;
                for (int i = 0; i < techUsed.Length; i++)
                {
                    if (techUsed[i])
                    {
                        ulong key = TechKeys[tech.Index] | TechKeys[i];
                        synergy += techData.GetCostAdjFactors(key, veh.TechnologyClass);
                    }
                }
            }
            else if (veh.ModelData.TechApplied[tech.Index])
            {   // technology is being adjusted for learning effect at the start of model year --
                //  1) consider only technologies that were applied by the model and aggregate
                //     synergy effect of this technology with all used technologies
                //  2) to prevent double counting, only scan technologies up to the index of this
                //     technology
                bool[] techUsed = veh.ModelData.TechUsed;
                for (int i = 0; i < tech.Index; i++)
                {
                    if (techUsed[i])
                    {
                        ulong key = TechKeys[tech.Index] | TechKeys[i];
                        synergy += techData.GetCostAdjFactors(key, veh.TechnologyClass);
                    }
                }
            }
            return synergy;
        }

        #endregion

        #region /* GetMaintenanceCost and GetRepairCost methods */

        /// <summary>
        /// Returns the maintenance cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the maintenance cost.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the maintenance cost of the <see cref="TechnologyInfo"/>.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <returns>The maintenance cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.</returns>
        public static double GetMaintenanceCost(TechnologyInfo tech, Vehicle veh, ModelYear year)
        {
            return GetEstimatedMaintenanceCost(tech, veh, year);
        }
        /// <summary>
        /// Returns the estimated maintenance cost.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the maintenance cost.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the maintenance cost of the <see cref="TechnologyInfo"/>.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <returns>The maintenance cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.</returns>
        static double GetEstimatedMaintenanceCost(TechnologyInfo tech, Vehicle veh, ModelYear year)
        {
            TechnologyCosts tc = tech.GetCosts(veh);
            //
            int yrIndex = Math.Min(tc.MaintenanceCostTable.Length - 1, year.Index);
            double cost = tc.MaintenanceCostTable[yrIndex];
            // return the estimated cost
            return cost;
        }

        /// <summary>
        /// Returns the repair cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the repair cost.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the repair cost of the <see cref="TechnologyInfo"/>.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <returns>The repair cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.</returns>
        public static double GetRepairCost(TechnologyInfo tech, Vehicle veh, ModelYear year)
        {
            return GetEstimatedRepairCost(tech, veh, year);
        }
        /// <summary>
        /// Returns the estimated repair cost.
        /// </summary>
        /// <param name="tech">The <see cref="TechnologyInfo"/> for which to estimate the repair cost.</param>
        /// <param name="veh">The <see cref="Vehicle"/> for which to estimate the repair cost of the <see cref="TechnologyInfo"/>.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <returns>The repair cost estimate of the specified <see cref="TechnologyInfo"/> for the given <see cref="Vehicle"/>.</returns>
        static double GetEstimatedRepairCost(TechnologyInfo tech, Vehicle veh, ModelYear year)
        {
            TechnologyAttributes ta = tech.GetAttributes(veh);
            TechnologyCosts      tc = tech.GetCosts     (veh);
            //
            int yrIndex = Math.Min(tc.RepairCostTable.Length - 1, year.Index);
            double cost = tc.RepairCostTable[yrIndex];
            //
            // consider special cases for some technologies
            if (TI.IsCostPerPound(tech.Index))
            {   // adjust cost based on vehicle's weight, using % of base veh's curb weight, absolute weight change, or none
                double deltaCW = GetDeltaWeight(tech, veh, year);
                cost *= deltaCW;
            }
            // return the estimated cost
            return cost;
        }

        #endregion

        #region /* GetCost/GetFC helper methods */

        static void GetTimeBasedTechYearsRange(ModelYear year, Vehicle veh, TechnologyInfo tech, out Vehicle ldr, out int ldrLastMY, out int ldrCurrMY)
        {
            int  tIndex           = tech.Index;
            bool isTiedToRedesign = TI.IsTiedToRedesign(tIndex);
            bool isTiedToRefresh  = TI.IsTiedToRefresh (tIndex);

            // find leader vehicle
            ldr = (TI.IsEngineLevel      (tIndex)) ? veh.Engine      .GetPlatformLeader() :
                  (TI.IsTransmissionLevel(tIndex)) ? veh.Transmission.GetPlatformLeader() :
                  (TI.IsPlatformLevel    (tIndex)) ? veh.Platform    .GetPlatformLeader() : veh;

            // find last model year when FCTimeBased technology is available for application on a vehicle
            // (this "last MY" represents the last year when the technology was applied on a vehicle)
            //  - if technology is redesing based, last available model year is the vehicle's preceding
            //    redesign year
            //  - if technology is refresh based, last available model year is the vehicle's preceding
            //    refresh or redesign year, whichever is greater
            //  - if technology is anytime (not tied to refresh or redesign), last available model year
            //    is the preceding analysis year
            // additionally, find two most recent model years when technology was available for application
            // on a leader vehicle, which represent "current applied year" and "last applied year" for this
            // technology, where:
            //  - ldrCurrMY must be less than or equal to the current analysis year
            //  - ldrLastMY must be less than or equal to the vehicle's "last MY"
            int vehLastMY;
            //
            if (isTiedToRedesign)
            {
                vehLastMY = veh.GetLastRedesign(year.Year    );
                ldrCurrMY = ldr.GetLastRedesign(year.Year + 1);
                ldrLastMY = ldr.GetLastRedesign(vehLastMY + 1);
            }
            else if (isTiedToRefresh)
            {
                vehLastMY = Math.Max(veh.GetLastRefresh(year.Year    ), veh.GetLastRedesign(year.Year    ));
                ldrCurrMY = Math.Max(ldr.GetLastRefresh(year.Year + 1), ldr.GetLastRedesign(year.Year + 1));
                ldrLastMY = Math.Max(ldr.GetLastRefresh(vehLastMY + 1), ldr.GetLastRedesign(vehLastMY + 1));
            }
            else
            {
                vehLastMY = year.Year - 1;
                ldrCurrMY = year.Year;
                ldrLastMY = vehLastMY;
            }
        }

        #endregion

        #endregion

        /// <summary>
        /// Checks whether a technology specfied by the given index can be backfilled on a vehicle, provided the list of
        /// technologies specified by techs is already selected to be applied and an array of technologies specified by techUsed
        /// is already being used by the vehicle.
        /// </summary>
        /// <param name="techIndex">The index of the technology to check whether it can be backfilled on a vehicle.</param>
        /// <param name="techs">A list of technologies selected to be applied.</param>
        /// <param name="techUsed">An array specifying technology used states for a vehicle.</param>
        /// <returns>true, if the technology specified by techIndex can be backfilled on a vehicle; false, otherwise.</returns>
        public static bool CanBackfill(int techIndex, List<TechnologyInfo> techs, bool[] techUsed)
        {
            for (int i = 0, count = techs.Count; i < count; i++)
            {   // check each technology selected to be applied, if it constraints with the techIndex technology
                if (!CanBackfillHelper(techIndex, techs[i].Index)) { return false; }
            }
            for (int i = 0; i < TI.TechnologyCount; i++)
            {   // check each technology already used by the vehicle, if it constraints with the techIndex technology
                if (techUsed[i] && !CanBackfillHelper(techIndex, i)) { return false; }
            }
            // at this point, the backfill conflicts for all technologies already added to the "to be applied" list
            // or used by the vehicle have been examined and no constraints have been found
            return true;
        }
        /// <summary>
        /// Checks whether a technology specfied by the given index can be backfilled on a vehicle, provided the list of
        /// technologies specified by techGroup and filtered by techKey is already selected to be applied and an array of
        /// technologies specified by techUsed is already being used by the vehicle.
        /// </summary>
        /// <param name="techIndex">The index of the technology to check whether it can be backfilled on a vehicle.</param>
        /// <param name="techGroup">A list of technologies in a group to consider for applicability.</param>
        /// <param name="techKey">A key (relative to techGroup) specifying the technologies selected for applicability.</param>
        /// <param name="techUsed">An array specifying technology used states for a vehicle.</param>
        /// <returns>true, if the technology specified by techIndex can be backfilled on a vehicle; false, otherwise.</returns>
        public static bool CanBackfill(int techIndex, List<TechnologyInfo> techGroup, ulong techKey, bool[] techUsed)
        {
            int trueTechIndex = techGroup[techIndex].Index;
            for (int i = 0, techCount = techGroup.Count; i < techCount; i++)
            {   // check each technology selected to be applied, if it constraints with the techIndex technology
                if ((techKey & TechKeys[i]) != 0)
                {   // this tech is selected to be applied on the current vehicle
                    if (!CanBackfillHelper(trueTechIndex, techGroup[i].Index)) { return false; }
                }
            }
            for (int i = 0; i < TI.TechnologyCount; i++)
            {   // check each technology already used by the vehicle, if it constraints with the techIndex technology
                if (techUsed[i] && !CanBackfillHelper(trueTechIndex, i)) { return false; }
            }
            // at this point, the backfill conflicts for all technologies already added to the "to be applied" list
            // or used by the vehicle have been examined and no constraints have been found
            return true;
        }
        static bool CanBackfillHelper(int techIndex, int selTechIndex)
        {
            // prevent self-backfilling (i.e. do not backfill this tech, since it is being examined)
            if (techIndex == selTechIndex) { return false; }
            int[] techDisallowBackfill = TechDisallowBackfill[selTechIndex];
            // check if the technology to be backfilled conflicts with the selected technology's disallow list
            for (int j = 0, disallowCount = techDisallowBackfill.Length; j < disallowCount; j++)
            {
                if (techIndex == techDisallowBackfill[j])
                {   // the technology cannot be backfilled because it conflicts with a technology
                    // on the disallow backfill list
                    return false;
                }
            }
            return true;
        }


        /// <summary>
        /// Determines whether a technology with the specified index is valid for applicability on a vehicle represented by the
        /// specified <see cref="Vehicle.CModelData"/>.
        /// </summary>
        /// <param name="modelYearData">An array of modeling data values, indexed by year, being analyzed with the current
        ///   scenario. This array is populated with each subsequent model year.</param>
        /// <param name="year">The model year in which to check technology applicability.</param>
        /// <param name="veh">The vehicle whose technology applicability to check.</param>
        /// <param name="techIdx">The index of the technology to check for applicability on a vehicle.</param>
        /// <param name="ignorePhaseIn">true, if phase-in caps on the specified technology should be ignored.</param>
        /// <param name="ignorePlatform">true to ignore platform dependency on engine, transmission, and platform level techs.</param>
        /// <param name="backfilling">true, if a technology with the specified index is being backfilled; false, otherwise.</param>
        /// <param name="lingering">When this method returns, specifies whether the technology is "lingering".</param>
        /// <param name="lingerYear">When this method returns, specifies the model year when the technology is lingering on the
        ///   vehicle, or null, if technology is not lingering on the vehicle.</param>
        /// <returns>true, if a technology with the specified index is valid for applicability on a vehicle; false, otherwise.</returns>
        public static bool IsTechValid(Industry[] modelYearData, ModelYear year, Vehicle veh, int techIdx,
            bool ignorePhaseIn, bool ignorePlatform, bool backfilling, out bool lingering, out ModelYear lingerYear)
        {
            lingering  = false;
            lingerYear = null;

            Manufacturer            mfr = veh.Manufacturer;
            Manufacturer.CModelData mmd = mfr.ModelData;
            Vehicle     .CModelData vmd = veh.ModelData;

            //----------------------------------------------------------------------------------//
            // The technology cannot be applied if:                                             //
            //  * it is not generally available on the vehicle (not compatible)                 //
            //  * it is already in use on the vehicle;                                          //
            //  * it has been ignored for the current model year AND it is not being backfilled //
            //    by another technology                                                         //
            //  * it has been exhausted on the manufacturer for the current model year AND it   //
            //    is not being backfilled by another technology AND the model is not ignoring   //
            //    phase-in caps                                                                 //
            //----------------------------------------------------------------------------------//
            if (!vmd.TechAvailable[techIdx]                  ||
                 vmd.TechUsed     [techIdx]                  ||
                (vmd.TechIgnored  [techIdx] && !backfilling) ||
                (mmd.TechExhausted[techIdx] && !backfilling && !ignorePhaseIn)) { return false; }

            //----------------------------------------------------------------------------------//
            // If technology is platform-specific, ensure it is applicable to the platform      //
            // leader as well.                                                                  //
            //----------------------------------------------------------------------------------//
            // obtain component to which this vehicle belongs
            Component platform = null;
            if (!ignorePlatform)
            {
                if      (TI.IsEngineLevel      (techIdx)) { platform = veh.Engine      ; }
                else if (TI.IsTransmissionLevel(techIdx)) { platform = veh.Transmission; }
                else if (TI.IsPlatformLevel    (techIdx)) { platform = veh.Platform    ; }
            }
            // obtain platform leader for the component
            Vehicle pLeader = (platform == null) ? null : platform.GetPlatformLeader();

            bool      pEnabled    = true;
            bool      pLingering  = false;
            ModelYear pLingerYear = null;
            if (pLeader != null)
            {   //------------------------------------------------------------------------------//
                // Determine technology applicability to the platform and the platform leader:  //
                // * if technology is already in use on the platform, it may not be enabled for //
                //   application on any vehicle (it must be inherited);                         //
                // * if vehicle is the platform leader, clear out pLeader variable and evaluate //
                //   applicability per below;                                                   //
                // * otherwise, determine if technology is applicable to the platform leader    //
                //------------------------------------------------------------------------------//
                if      (platform.ModelData.TechUsed[techIdx]) { pEnabled = false; }
                else if (veh == pLeader) { pLeader = null; }
                else
                {
                    pEnabled = IsTechValid(modelYearData, year, pLeader, techIdx, ignorePhaseIn, ignorePlatform, backfilling,
                        out pLingering, out pLingerYear);
                }
            }
            if (!pEnabled) { return false; }

            // if platform leader is lingering, obtain vehicle linger states closest to platform's linger year
            // otherwise, obtain the most recent vehicle linger states
            ModelYear vehTechLingerYear = null;
            bool[]    vehTechLingering  = null;
            if (pLingering)
            {
                for (int i = 0; i < vmd.TechLingerYear.Count; i++)
                {
                    if (vmd.TechLingerYear[i].Year >= pLingerYear.Year)
                    {
                        vehTechLingerYear = vmd.TechLingerYear[i];
                    }
                }
            }
            else
            {
                int lastLinger = vmd.TechLingerYear.Count - 1;
                if (lastLinger != -1)
                {
                    vehTechLingerYear = vmd.TechLingerYear[lastLinger];
                }
            }
            if (vehTechLingerYear != null)
            {
                Manufacturer lMfr = modelYearData[vehTechLingerYear.Index].Manufacturers[mfr.Index];
                Vehicle      lVeh = lMfr.Vehicles.Find((query) => { return (query.Description.Code == veh.Description.Code); });
                vehTechLingering  = lVeh.ModelData.TechEnabled;
            }

            //----------------------------------------------------------------------------------//
            // Evaluate technology application on the vehicle.                                  //
            //----------------------------------------------------------------------------------//
            if (vmd.TechEnabled[techIdx] && !pLingering)
            {   // the technology is enabled for the current model year on this vehicle as well as the platform leader
                return true;
            }
            if (vmd.TechEnabled[techIdx] && (vehTechLingerYear == null || vehTechLingerYear.Year < pLingerYear.Year))
            {   // the technology is enabled for the current model year on this vehicle, but lingering on the platform leader --
                // additionally, vehicle's selected linger year occurs before the platform's linger year
                return true;
            }
            if (vehTechLingering != null && vehTechLingering[techIdx] && (pLeader == null || pLingering))
            {   // the technology was not enabled for the current model year but it is lingering from the previous one
                // (alternatively, technology may have been enabled on the vehicle but the platfrom was lingering --
                //  in this case, the vehicle's linger state should still be considered)
                //
                // a vehicle is only allowed to consider technology linger state if the platform leader is also lingering
                // or if the vehicle does not have a platform leader (a platform leader does not exists if the tech is not
                // platfrom/engine/transmission specific, or the model is not operating in tech-inheriting mode)
                //
                // when lingering, need to additionally consider exhausted setting of a technology as follows:
                //  * if platform leader exists, check exhausted state during the platform leader's linger year
                //  * if there is no platform leader, check exhausted state on the vehicle's linger year
                int                     lYr  = (pLingering) ? pLingerYear.Index : vehTechLingerYear.Index;
                Manufacturer.CModelData lMMD = modelYearData[lYr].Manufacturers[mfr.Index].ModelData;
                if (backfilling || ignorePhaseIn || !lMMD.TechExhausted[techIdx])
                {   // if technology is still applicable in the linger year, ensure that the linger year of the vehicle
                    // occurs on or after the platform's linger year
                    // alternatively, if the platform leader does not exist, the technology is automatically assumed to be
                    // lingering on the vehicle
                    if (pLeader == null || vehTechLingerYear.Year >= pLingerYear.Year)
                    {
                        lingering  = true;
                        lingerYear = vehTechLingerYear;
                        return true;
                    }
                }
            }

            // return false if the tech is not enabled in the current or lingering year
            return false;
        }

        /// <summary>
        /// Installs the provided technology on a specified vehicle, for the model year being analyzed, using the given modeling
        /// settings.
        /// </summary>
        /// <param name="veh">The vehicle for which to install a technology.</param>
        /// <param name="tech">The technology to install on a vehicle.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="yearOfInitialApplication">If the technology is applied due to multiyear, specifies the initial year when
        ///   the technology was installed on a vehicle.</param>
        /// <param name="isInherited">true, if technology was applied to a vehicle because of inheriting; false, otherwise.</param>
        /// <param name="settings">The modeling settings used for modeling.</param>
        public static void InstallTechnologyOnVehicle(Vehicle veh, TechnologyInfo tech, ModelYear year, int yearOfInitialApplication,
            bool isInherited, ModelingSettings settings)
        {
            Vehicle     .CDescription vd  =                                     veh             .Description;
            Engine      .CDescription ed  = (veh.Engine       == null) ? null : veh.Engine      .Description;
            Transmission.CDescription td  = (veh.Transmission == null) ? null : veh.Transmission.Description;
            Vehicle     .CModelData   vmd =                                     veh             .ModelData;
            Engine      .CModelData   emd = (veh.Engine       == null) ? null : veh.Engine      .ModelData;
            Transmission.CModelData   tmd = (veh.Transmission == null) ? null : veh.Transmission.ModelData;
            Platform    .CModelData   pmd = (veh.Platform     == null) ? null : veh.Platform    .ModelData;

            // get the tech attributes
            TechnologyAttributes attr = tech.GetAttributes(veh);

            // apply the technology -- set tech-used and tech-applied states on the vehicle
            vmd.TechUsed       [tech.Index] = true;
            vmd.TechInherited  [tech.Index] = isInherited;
            vmd.TechApplied    [tech.Index] = true;
            vmd.TechAppliedYear[tech.Index] = yearOfInitialApplication;

            // set tech-used and tech-applied states on the vehicle's engine, transmission, and platform
            if (TI.IsEngineLevel(tech.Index) && veh.IsEngineLeader() && emd != null && !emd.TechUsed[tech.Index])
            {
                emd.TechUsed       [tech.Index] = true;
                emd.TechApplied    [tech.Index] = true;
                emd.TechAppliedYear[tech.Index] = yearOfInitialApplication;
            }
            if (TI.IsTransmissionLevel(tech.Index) && veh.IsTransmissionLeader() && tmd != null && !tmd.TechUsed[tech.Index])
            {
                tmd.TechUsed       [tech.Index] = true;
                tmd.TechApplied    [tech.Index] = true;
                tmd.TechAppliedYear[tech.Index] = yearOfInitialApplication;
            }
            if (TI.IsPlatformLevel(tech.Index) && veh.IsPlatformLeader() && pmd != null && !pmd.TechUsed[tech.Index])
            {
                pmd.TechUsed       [tech.Index] = true;
                pmd.TechApplied    [tech.Index] = true;
                pmd.TechAppliedYear[tech.Index] = yearOfInitialApplication;
            }

            // recalculate vehicle-specific FCAdjKey
            // (full recalc is requied since we now allow out-of-order engine techs)
            InitializeFCAdjKey(veh);
            //// update vehicle-specific FCAdjKey
            //vmd.FCAdjKey = GetFCAdjKey(veh, tech);

            // --------------------------------------------------
            // update vehicle characteristics
            // --------------------------------------------------
            // engine technologies
            if      (TI.IsConversionToCNG   (tech.Index)) { ed.Fuel = FuelType.CNG; }
            else if (TI.IsConversionToDiesel(tech.Index)) { ed.Fuel = FuelType.Diesel; }
            // transmission technologies
            else if (TI.IsManualTransmissionPath(tech.Index))
            {
                td.Type = "M";
                td.NumForwardGears =
                    (tech.Index == TI.MT7) ? 7 :
                    (tech.Index == TI.MT6) ? 6 :
                    (tech.Index == TI.MT5) ? 5 : 4;
            }
            else if (tech.Index == TI.CVT)
            {
                td.Type = "CVT";
                td.NumForwardGears = 1;
            }
            else if (tech.Index == TI.DCT6 || tech.Index == TI.DCT8)
            {
                td.Type = "S";
                td.NumForwardGears = (tech.Index == TI.DCT6) ? 6 : 8;
            }
            else if (TI.IsAutoTransmissionPath(tech.Index))
            {
                td.Type = "A";
                td.NumForwardGears =
                    (tech.Index == TI.AT8) ? 8 :
                    (tech.Index == TI.AT6) ? 6 :
                    (tech.Index == TI.AT5) ? 5 : 4;
            }
            // hybrid technologies
            else if (TI.IsConversionToMHEV(tech.Index)) { veh.HEVType = HEVType.MildHybrid; }
            else if (TI.IsConversionToSHEV(tech.Index))
            {
                veh.HEVType = HEVType.StrongHybrid;
                if (tech.Index == TI.SHEVP2)
                {   // SHEVP2 uses DCT6, while all other transmission techs should be set to superseded
                    for (int i = 0; i < TI.TechnologyCount; i++)
                    {
                        if (TI.IsAutoTransmissionPath(i) && vmd.TechUsed[i])
                        {
                            vmd.TechSuperseded    [i] = true;
                            vmd.TechSupersededYear[i] = yearOfInitialApplication;
                        }
                    }
                    //
                    vmd.TechUsed          [TI.DCT6] = true;
                    vmd.TechInherited     [TI.DCT6] = isInherited;
                    vmd.TechApplied       [TI.DCT6] = true;
                    vmd.TechAppliedYear   [TI.DCT6] = yearOfInitialApplication;
                    vmd.TechSuperseded    [TI.DCT6] = false;
                    vmd.TechSupersededYear[TI.DCT6] = 0;
                }
                else if (tech.Index == TI.SHEVPS)
                {
                    veh.RemoveEngine();
                }
                veh.RemoveTransmission();
            }
            else if (TI.IsConversionToPHEV(tech.Index))
            {
                veh.HEVType      = HEVType.PlugInHybrid;
                vmd.ZEVCredits   = tech.ZEVCredits;
                vd .VehicleRange = attr.ElectricRange;
                veh.RemoveTransmission();
                veh.RemoveEngine();
            }
            else if (TI.IsConversionToEV(tech.Index))
            {
                veh.HEVType      = HEVType.PureElectric;
                vmd.ZEVCredits   = tech.ZEVCredits;
                vd .VehicleRange = attr.ElectricRange;
                veh.RemoveTransmission();
                veh.RemoveEngine();
            }
            else if (TI.IsConversionToFCV(tech.Index))
            {
                veh.HEVType      = HEVType.FuelCell;
                vmd.ZEVCredits   = tech.ZEVCredits;
                vd .VehicleRange = attr.ElectricRange;
                veh.RemoveTransmission();
                veh.RemoveEngine();
            }

            // update the enabled state of the technology, ensuring that proper techs get enabled or disabled
            // based on technology dependencies
            TechnologyApplicability.Enable(veh, year, tech.Index, settings);
        }


        /// <summary>
        /// Determines whether the specified technology has reached its phase-in cap, for the specified model year, and exhausts
        /// its applicability for the provided manufacturer if it has.
        /// </summary>
        /// <param name="settings"></param>
        /// <param name="mmd">The modeling data of a manufacturer for which to update the tech-exhausted flags if technology
        ///   phase-in has been exhausted.</param>
        /// <param name="tech">The technology for which to check the phase-in.</param>
        /// <param name="year">The current model year being analyzed.</param>
        /// <param name="sYrIndex">The index of the model year when modeling began.</param>
        public static void CheckPhaseIn(ModelingSettings settings, Manufacturer.CModelData mmd, TechnologyInfo tech, ModelYear year,
            int sYrIndex)
        {
            // do not check phase-in for technologies that have already been exhausted
            if (mmd.TechExhausted[tech.Index]) { return; }

            // check for phase-in pairs -- all pairs should use a single phase-in value
            int[] phaseInPairs = TI.GetSharedPhaseInTechs(tech.Index);
            //
            if (phaseInPairs == null)
            {   // the current tech is not part of a phase-in pair
                double techSales = mmd.TechAppliedSales[tech.Index].Total;
                double mfrSales  = mmd.Sales.Total;
                if (mfrSales > 0 && techSales >= mfrSales * tech.GetPhaseIn(year.Index, sYrIndex))
                {   // exhaust phase-in of the tech
                    if (tech.Index != -1) { mmd.TechExhausted[tech.Index] = true; }
                }
            }
            else
            {   // compute the tech sales (for all techs that are mutually phased-in) based on "applied" or "used" sales
                double techSales = 0D;
                for (int i = 0; i < phaseInPairs.Length; i++)
                {
                    if (phaseInPairs[i] != -1) { techSales += mmd.TechAppliedSales[phaseInPairs[i]].Total; }
                }
                // check if the tech pairs are phased-in, and "exhaust" as necessary
                double mfrSales = mmd.Sales.Total;
                if (mfrSales > 0 && techSales >= mfrSales * tech.GetPhaseIn(year.Index, sYrIndex))
                {   // exhaust phase-in of the tech and all its pairs (if any)
                    for (int i = 0; i < phaseInPairs.Length; i++)
                    {
                        if (phaseInPairs[i] != -1) { mmd.TechExhausted[phaseInPairs[i]] = true; }
                    }
                }
            }
        }

        /// <summary>
        /// Initialize a technology key used for looking up fuel consumption adjustment factors for the specified vehicle.
        /// </summary>
        /// <param name="veh">The vehicle for which to initialize the technology key.</param>
        public static void InitializeFCAdjKey(Vehicle veh)
        {
            Vehicle.CModelData vmd = veh.ModelData;

            // compute new key from used technologies
            vmd.FCAdjKey = KeyHelper.BuildKey(vmd.TechUsed);

            // build a "remove-key" (all techs to exclude from FCAdjKey)
            ulong fcAdjRemoveKey = 0UL;
            for (int i = 0; i < vmd.TechUsed.Length; i++)
            {
                if (vmd.TechUsed[i])
                {
                    if (TI.IsFCAdjKey(i))
                    {   // remove technologies replaced by this one
                        fcAdjRemoveKey |= KeyHelper.BuildKey(TechRemoveForFCAdj[i]);
                    }
                    else
                    {   // remove used technologies that do not affect FCAdjKey
                        fcAdjRemoveKey |= KeyHelper.Keys[i];
                    }
                }
            }

            // update FCAdjKey
            vmd.FCAdjKey &= (~fcAdjRemoveKey);
        }
        static ulong GetFCAdjKey(Vehicle veh, TechnologyInfo tech)
        {
            ulong fcAdjKey = veh.ModelData.FCAdjKey;
            if (TI.IsFCAdjKey(tech.Index))
            {
                fcAdjKey = veh.ModelData.FCAdjKey & (~KeyHelper.BuildKey(TechRemoveForFCAdj[tech.Index])) | KeyHelper.Keys[tech.Index];
            }
            return fcAdjKey;
        }

        #endregion

        #region /*** Variables ***/

        /// <summary>Defines glider shares for each technology class.</summary>
        static double[] GliderShare = new double[] { 0.6444, 0.6669, 0.6701, 0.6606, 0.6647, 1, 1 };

        /// <summary>Defines superseding technology arrays (read-only values), initialized in the constructor.  The outer array
        ///   represents the index of a technology affected, while the inner array represents zero or more indices of
        ///   technologies superseded by the affected technology.</summary>
        static readonly int[][] TechSupersedes;
        /// <summary>Defines soft-superseding technology arrays (read-only values), initialized in the constructor.  The outer
        ///   array represents the index of a technology affected, while the inner array represents zero or more indices of
        ///   technologies soft-superseded by the affected technology.  Soft-superseding applies to early replacement penalty
        ///   costs and does not affect technology application or penetration rates.</summary>
        static readonly int[][] TechSoftSupersedes;
        /// <summary>This variable is a combination of <see cref="TechSupersedes"/> and <see cref="TechSoftSupersedes"/>.</summary>
        static readonly int[][] TechSupersedes2;

        /// <summary>Defines backfill disallow technology arrays (read-only values), initialized in the constructor.  The outer
        ///   array represents the index of a technology affected, while the inner array represents zero or more indices of
        ///   technologies that conflict with the affected technology and are disallowed from being backfilled.</summary>
        static readonly int[][] TechDisallowBackfill;

        /// <summary>Defines technologies will be removed from the vehicle-level "fcAdjKey" before the defining technology is
        ///   added. The "fcAdjKey" is used for looking up the FC adjustment factors in the FC adjustments table. The outer array
        ///   represents the index of a technology affected, while the inner array represents zero or more indices of technologies
        ///   removed by the affected technology.</summary>
        static readonly int[][] TechRemoveForFCAdj;

        /// <summary>Defines technology keys for use during compliance finding as:  TechKeys[i] = Math.Pow(2, i).  The TechKeys
        ///   array contains 63 elements.</summary>
        public static readonly ulong[] TechKeys;

        /// <summary>Defines a 0-based array of regulatory classes from the <see cref="RegulatoryClass"/>, where: index 0 is
        ///   <see cref="RegulatoryClass.PassengerCar"/>, index 1 is <see cref="RegulatoryClass.LightTruck"/>, and index 2 is
        ///   <see cref="RegulatoryClass.LightTruck2b3"/>.</summary>
        public static RegulatoryClass[] RegClasses;

        #endregion

    }
}
