using System;
using System.Text;
using Volpe.Cafe.Generic;

namespace Volpe.Cafe.Data
{
	/// <summary>
	/// Provides an object to store model data by fuel type.
	/// </summary>
    [Serializable]
    public struct FuelValue
	{

        #region /*** Constructors ***/

        static FuelValue()
        {
            FuelValue.NaN  = new FuelValue(double.NaN);
            FuelValue.Zero = new FuelValue(0);
            FuelValue.One  = new FuelValue(1);
        }
        FuelValue(double value) : this(value, value, value, value, value, value) { }
        FuelValue(double gasoline, double ethanol85, double diesel, double electricity, double hydrogen, double cng) : this()
        {
            this.Gasoline    = gasoline;
            this.Ethanol85   = ethanol85;
            this.Diesel      = diesel;
            this.Electricity = electricity;
            this.Hydrogen    = hydrogen;
            this.CNG         = cng;
        }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the sum of the two specified <see cref="FuelValue"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the two specified <see cref="FuelValue"/> values.</returns>
        public static FuelValue operator +(FuelValue value1, FuelValue value2)
        {
            return new FuelValue(value1.Gasoline    + value2.Gasoline,
                                 value1.Ethanol85   + value2.Ethanol85,
                                 value1.Diesel      + value2.Diesel,
                                 value1.Electricity + value2.Electricity,
                                 value1.Hydrogen    + value2.Hydrogen,
                                 value1.CNG         + value2.CNG);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the sum of the specified <see cref="FuelValue"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="FuelValue"/> value and a <see cref="Double"/> value.</returns>
        public static FuelValue operator +(FuelValue value1, double value2)
        {
            return value1 + new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the sum of the specified <see cref="FuelValue"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="FuelValue"/> value and a <see cref="Double"/> value.</returns>
        public static FuelValue operator +(double value1, FuelValue value2)
        {
            return new FuelValue(value1) + value2;
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the sum of the specified <see cref="FuelValue"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="FuelValue"/> value and an <see cref="Int32"/> value.</returns>
        public static FuelValue operator +(FuelValue value1, int value2)
        {
            return value1 + new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the sum of the specified <see cref="FuelValue"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="FuelValue"/> value and an <see cref="Int32"/> value.</returns>
        public static FuelValue operator +(int value1, FuelValue value2)
        {
            return new FuelValue(value1) + value2;
        }

        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static FuelValue operator -(FuelValue value1, FuelValue value2)
        {
            return new FuelValue(value1.Gasoline    - value2.Gasoline,
                                 value1.Ethanol85   - value2.Ethanol85,
                                 value1.Diesel      - value2.Diesel,
                                 value1.Electricity - value2.Electricity,
                                 value1.Hydrogen    - value2.Hydrogen,
                                 value1.CNG         - value2.CNG);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static FuelValue operator -(FuelValue value1, double value2)
        {
            return value1 - new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static FuelValue operator -(double value1, FuelValue value2)
        {
            return new FuelValue(value1) - value2;
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static FuelValue operator -(FuelValue value1, int value2)
        {
            return value1 - new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static FuelValue operator -(int value1, FuelValue value2)
        {
            return new FuelValue(value1) - value2;
        }

        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the product of the two specified <see cref="FuelValue"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the two specified <see cref="FuelValue"/> values.</returns>
        public static FuelValue operator *(FuelValue value1, FuelValue value2)
        {
            return new FuelValue(value1.Gasoline    * value2.Gasoline,
                                 value1.Ethanol85   * value2.Ethanol85,
                                 value1.Diesel      * value2.Diesel,
                                 value1.Electricity * value2.Electricity,
                                 value1.Hydrogen    * value2.Hydrogen,
                                 value1.CNG         * value2.CNG);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the product of the specified <see cref="FuelValue"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="FuelValue"/> value and a <see cref="Double"/> value.</returns>
        public static FuelValue operator *(FuelValue value1, double value2)
        {
            return value1 * new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the product of the specified <see cref="FuelValue"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="FuelValue"/> value and a <see cref="Double"/> value.</returns>
        public static FuelValue operator *(double value1, FuelValue value2)
        {
            return new FuelValue(value1) * value2;
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the product of the specified <see cref="FuelValue"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="FuelValue"/> value and an <see cref="Int32"/> value.</returns>
        public static FuelValue operator *(FuelValue value1, int value2)
        {
            return value1 * new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the product of the specified <see cref="FuelValue"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="FuelValue"/> value and an <see cref="Int32"/> value.</returns>
        public static FuelValue operator *(int value1, FuelValue value2)
        {
            return new FuelValue(value1) * value2;
        }

        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static FuelValue operator /(FuelValue value1, FuelValue value2)
        {
            return new FuelValue(value1.Gasoline    / value2.Gasoline,
                                 value1.Ethanol85   / value2.Ethanol85,
                                 value1.Diesel      / value2.Diesel,
                                 value1.Electricity / value2.Electricity,
                                 value1.Hydrogen    / value2.Hydrogen,
                                 value1.CNG         / value2.CNG);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static FuelValue operator /(FuelValue value1, double value2)
        {
            return value1 / new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static FuelValue operator /(double value1, FuelValue value2)
        {
            return new FuelValue(value1) / value2;
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static FuelValue operator /(FuelValue value1, int value2)
        {
            return value1 / new FuelValue(value2);
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static FuelValue operator /(int value1, FuelValue value2)
        {
            return new FuelValue(value1) / value2;
        }

        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value, which is the copy of the specified <see cref="FuelValue"/> value.
        /// </summary>
        /// <param name="value">The value to copy.</param>
        /// <returns>A copy of the specified <see cref="FuelValue"/> value.</returns>
        public static FuelValue operator +(FuelValue value)
        {
            return FuelValue.Zero + value;
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value, which is the negative of the specified <see cref="FuelValue"/> value.
        /// </summary>
        /// <param name="value">The value to negate.</param>
        /// <returns>The specified <see cref="FuelValue"/> value negated.</returns>
        public static FuelValue operator -(FuelValue value)
        {
            return FuelValue.Zero - value;
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value, which is one more than the specified value (increment).
        /// </summary>
        /// <param name="value">The value to increment by one.</param>
        /// <returns>The specified value, incremented by one.</returns>
        public static FuelValue operator ++(FuelValue value)
        {
            return value + FuelValue.One;
        }
        /// <summary>
        /// Returns a new <see cref="FuelValue"/> value, which is one less than the specified value (decrement).
        /// </summary>
        /// <param name="value">The value to decrement by one.</param>
        /// <returns>The specified value, decremented by one.</returns>
        public static FuelValue operator --(FuelValue value)
        {
            return value - FuelValue.One;
        }
        /// <summary>
        /// Determines whether the specified <see cref="FuelValue"/> values are equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="FuelValue"/> values are equal; false, otherwise.</returns>
        public static bool operator ==(FuelValue value1, FuelValue value2)
        {
            return value1.Equals(value2);
        }
        /// <summary>
        /// Determines whether the specified <see cref="FuelValue"/> values are not equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="FuelValue"/> values are not equal; false, otherwise.</returns>
        public static bool operator !=(FuelValue value1, FuelValue value2)
        {
            return !value1.Equals(value2);
        }

        #endregion

        #region /* Overriden from object */

        /// <summary>
        /// Returns the string representation of this <see cref="FuelValue"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="FuelValue"/> instance.</returns>
        public override string ToString()
        {
            return "{Gasoline="    + this.Gasoline   .ToString() +
                  ", Ethanol85="   + this.Ethanol85  .ToString() +
                  ", Diesel="      + this.Diesel     .ToString() +
                  ", Electricity=" + this.Electricity.ToString() +
                  ", Hydrogen="    + this.Hydrogen   .ToString() +
                  ", CNG="         + this.CNG        .ToString() + "}";
        }
        /// <summary>
        /// Returns the string representation of this <see cref="FuelValue"/> instance, using the specified format for all elements.
        /// </summary>
        /// <param name="format">A format string.</param>
        /// <returns>The string representation of the <see cref="FuelValue"/> instance, using the specified format.</returns>
        public string ToString(string format)
        {
            return this.ToString(format, false);
        }
        /// <summary>
        /// Returns the string representation of this <see cref="FuelValue"/> instance, whose elements are rounded with the
        /// specified precision.
        /// </summary>
        /// <param name="digits">The rounding precision; e.g. number of digits after the period.</param>
        /// <returns>The string representation of the <see cref="FuelValue"/> instance.</returns>
        public string ToString(int digits)
        {
            return "{Gasoline="    + Math.Round(this.Gasoline   , digits).ToString() +
                  ", Ethanol85="   + Math.Round(this.Ethanol85  , digits).ToString() +
                  ", Diesel="      + Math.Round(this.Diesel     , digits).ToString() +
                  ", Electricity=" + Math.Round(this.Electricity, digits).ToString() +
                  ", Hydrogen="    + Math.Round(this.Hydrogen   , digits).ToString() +
                  ", CNG="         + Math.Round(this.CNG        , digits).ToString() + "}";
        }
        /// <summary>
        /// Returns the string representation of this <see cref="FuelValue"/> instance, using the specified format for all elements,
        /// optionally excluding fuel types with zero value.
        /// </summary>
        /// <param name="format">A format string.</param>
        /// <param name="nonZeroOnly">true, to exclude fuel types with zero value; false, to include all fuel types.</param>
        /// <returns>The string representation of the <see cref="FuelValue"/> instance, using the specified format.</returns>
        public string ToString(string format, bool nonZeroOnly)
        {
            return this.ToString(format, nonZeroOnly, false);
        }
        /// <summary>
        /// Returns the string representation of this <see cref="FuelValue"/> instance, using the specified format for all elements,
        /// optionally excluding fuel types with zero value.
        /// </summary>
        /// <param name="format">A format string.</param>
        /// <param name="nonZeroOnly">true, to exclude fuel types with zero value; false, to include all fuel types.</param>
        /// <param name="acronyms">true, to use fuel type acronyms; false, to use full fuel type names.</param>
        /// <returns>The string representation of the <see cref="FuelValue"/> instance, using the specified format.</returns>
        public string ToString(string format, bool nonZeroOnly, bool acronyms)
        {
            StringBuilder value = new StringBuilder();
            value.Append('{');
            //
            for (int i = 0; i < FTValue<object>.Classes.Length; i++)
            {
                double d = this[FTValue<object>.Classes[i]];
                if (d != 0 || !nonZeroOnly)
                {
                    if (acronyms)
                    {
                        value.Append(FTValue<object>.Acronyms[i]);
                    }
                    else
                    {
                        value.Append(FTValue<object>.Classes[i].ToString());
                    }
                    value.Append('=');
                    value.Append(d.ToString(format));
                    value.Append(',');
                    value.Append(' ');
                }
            }
            if (value.Length > 1) { value.Length -= 2; }
            value.Append('}');
            //
            return value.ToString();
        }

        /// <summary>
        /// Serves as a hash function for a particular type, suitable for use in hashing algorithms and data structures like a
        /// hash table.
        /// </summary>
        /// <returns>A hash code for the current <see cref="FuelValue"/>.</returns>
        public override int GetHashCode()
        {
            return (this.Gasoline   .GetHashCode() ^
                    this.Ethanol85  .GetHashCode() ^
                    this.Diesel     .GetHashCode() ^
                    this.Electricity.GetHashCode() ^
                    this.Hydrogen   .GetHashCode() ^
                    this.CNG        .GetHashCode());
        }
        /// <summary>
        /// Determines whether the specified <see cref="Object"/> is equal to the current <see cref="FuelValue"/> value.
        /// </summary>
        /// <param name="obj">The <see cref="Object"/> to compare with the current <see cref="FuelValue"/> value.</param>
        /// <returns>true, if the specified <see cref="Object"/> is equal to the current <see cref="FuelValue"/> value; false,
        ///   otherwise.</returns>
        public override bool Equals(object obj)
        {
            return (obj is FuelValue) ? this.Equals((FuelValue)obj) : base.Equals(obj);
        }
        /// <summary>
        /// Determines whether the specified <see cref="FuelValue"/> value is equal to the current <see cref="FuelValue"/> value.
        /// </summary>
        /// <param name="value">The <see cref="FuelValue"/> value to compare with the current <see cref="FuelValue"/> value.</param>
        /// <returns>true, if the specified <see cref="FuelValue"/> value is equal to the current <see cref="FuelValue"/> value;
        ///   false, otherwise.</returns>
        public bool Equals(FuelValue value)
        {
            return (this.Gasoline    == value.Gasoline    &&
                    this.Ethanol85   == value.Ethanol85   &&
                    this.Diesel      == value.Diesel      &&
                    this.Electricity == value.Electricity &&
                    this.Hydrogen    == value.Hydrogen    &&
                    this.CNG         == value.CNG);
        }

        #endregion

        /// <summary>
        /// Returns the sum of the specified values.
        /// </summary>
        /// <param name="values">One or more values to sum.</param>
        /// <returns>The sum of the specified values.</returns>
        public static FuelValue Sum(params FuelValue[] values)
        {
            FuelValue sum = new FuelValue();
            for (int i = 0; i < values.Length; i++)
            {
                sum += values[i];
            }
            return sum;
        }

        /// <summary>
        /// Resets all members of this <see cref="FuelValue"/> instance to their default values (0.0).
        /// </summary>
        public void Clear()
        {
            this.Gasoline    =
            this.Ethanol85   =
            this.Diesel      =
            this.Electricity =
            this.Hydrogen    =
            this.CNG         = 0;
        }

        //----- methods for computing averages -----
        /// <summary>
        /// Computes the arithmetic mean of this fuel value.
        /// </summary>
        public double ArithmeticMean()
        {
            return this.ArithmeticMean(FuelValue.One);
        }
        /// <summary>
        /// Computes the weighted arithmetic mean of this fuel value.
        /// </summary>
        /// <param name="weight">The weight value to use for averaging.</param>
        public double ArithmeticMean(FuelValue weight)
        {
            return (this * weight).Total / weight.Total;
        }
        /// <summary>
        /// Computes the geometric mean of this fuel value.
        /// </summary>
        public double GeometricMean()
        {
            return this.GeometricMean(FuelValue.One);
        }
        /// <summary>
        /// Computes the weighted geometric mean of this fuel value.
        /// </summary>
        /// <param name="weight">The weight value to use for averaging.</param>
        public double GeometricMean(FuelValue weight)
        {
            double d = 0;
            //
            foreach (FuelType fuel in FTValue<object>.Classes)
            {
                if (this[fuel] != 0 && !double.IsNaN(this[fuel]) && !double.IsInfinity(this[fuel]))
                {
                    d += weight[fuel] * Math.Log(this[fuel]);
                }
            }
            //
            return Math.Exp(d / weight.Total);
        }
        /// <summary>
        /// Computes the harmonic mean of this fuel value.
        /// </summary>
        public double HarmonicMean()
        {
            return this.HarmonicMean(FuelValue.One);
        }
        /// <summary>
        /// Computes the weighted harmonic mean of this fuel value.
        /// </summary>
        /// <param name="weight">The weight value to use for averaging.</param>
        public double HarmonicMean(FuelValue weight)
        {
            double d = 0;
            //
            foreach (FuelType fuel in FTValue<object>.Classes)
            {
                if (this[fuel] != 0 && !double.IsNaN(this[fuel]) && !double.IsInfinity(this[fuel]))
                {
                    d += weight[fuel] / this[fuel];
                }
            }
            //
            return (d == 0) ? 0 : weight.Total / d;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets or sets a fuel value component for the specified <see cref="FuelType"/>. When getting,
        ///   <see cref="Volpe.Cafe.FuelType.All"/> may be used to obtain a total sum of all fuel types; multiple
        ///   <see cref="FuelType"/> values may also be combined to obtain a sum of those fuel types.</summary>
        public double this[FuelType fuelType]
        {
            get
            {
                if      (fuelType == FuelType.Gasoline   ) { return this.Gasoline   ; }
                else if (fuelType == FuelType.Ethanol85  ) { return this.Ethanol85  ; }
                else if (fuelType == FuelType.Diesel     ) { return this.Diesel     ; }
                else if (fuelType == FuelType.Electricity) { return this.Electricity; }
                else if (fuelType == FuelType.Hydrogen   ) { return this.Hydrogen   ; }
                else if (fuelType == FuelType.CNG        ) { return this.CNG        ; }
                else if (fuelType == FuelType.All        ) { return this.Total      ; }
                else
                {   // check for individual combination of fuel types
                    double value = 0;
                    //
                    if ((fuelType & FuelType.Gasoline   ) == FuelType.Gasoline   ) { value += this.Gasoline;    }
                    if ((fuelType & FuelType.Ethanol85  ) == FuelType.Ethanol85  ) { value += this.Ethanol85;   }
                    if ((fuelType & FuelType.Diesel     ) == FuelType.Diesel     ) { value += this.Diesel;      }
                    if ((fuelType & FuelType.Electricity) == FuelType.Electricity) { value += this.Electricity; }
                    if ((fuelType & FuelType.Hydrogen   ) == FuelType.Hydrogen   ) { value += this.Hydrogen;    }
                    if ((fuelType & FuelType.CNG        ) == FuelType.CNG        ) { value += this.CNG;         }
                    //
                    return value;
                }
            }
            set
            {
                if      (fuelType == FuelType.Gasoline   ) { this.Gasoline    = value; }
                else if (fuelType == FuelType.Ethanol85  ) { this.Ethanol85   = value; }
                else if (fuelType == FuelType.Diesel     ) { this.Diesel      = value; }
                else if (fuelType == FuelType.Electricity) { this.Electricity = value; }
                else if (fuelType == FuelType.Hydrogen   ) { this.Hydrogen    = value; }
                else if (fuelType == FuelType.CNG        ) { this.CNG         = value; }
            }
        }

        /// <summary>Gets the Gasoline component of this fuel value.</summary>
        public double Gasoline { get; private set; }
        /// <summary>Gets the Ethanol-85 component of this fuel value.</summary>
        public double Ethanol85 { get; private set; }
        /// <summary>Gets the Diesel component of this fuel value.</summary>
        public double Diesel { get; private set; }
        /// <summary>Gets the Electricity component of this fuel value.</summary>
        public double Electricity { get; private set; }
        /// <summary>Gets the Hydrogen component of this fuel value.</summary>
        public double Hydrogen { get; private set; }
        /// <summary>Gets the CNG component of this fuel value.</summary>
        public double CNG { get; private set; }

        /// <summary>Gets the sum of all components of this fuel value.</summary>
        public double Total
        {
            get
            {
                // check for undefined numbers first
                if (this.IsNaN             ) { return double.NaN;              }
                if (this.IsPositiveInfinity) { return double.PositiveInfinity; }
                if (this.IsNegativeInfinity) { return double.NegativeInfinity; }
                //
                // compute total
                double total = 0;
                //
                if (!double.IsNaN(this.Gasoline   ) && !double.IsInfinity(this.Gasoline   )) { total += this.Gasoline   ; }
                if (!double.IsNaN(this.Ethanol85  ) && !double.IsInfinity(this.Ethanol85  )) { total += this.Ethanol85  ; }
                if (!double.IsNaN(this.Diesel     ) && !double.IsInfinity(this.Diesel     )) { total += this.Diesel     ; }
                if (!double.IsNaN(this.Electricity) && !double.IsInfinity(this.Electricity)) { total += this.Electricity; }
                if (!double.IsNaN(this.Hydrogen   ) && !double.IsInfinity(this.Hydrogen   )) { total += this.Hydrogen   ; }
                if (!double.IsNaN(this.CNG        ) && !double.IsInfinity(this.CNG        )) { total += this.CNG        ; }
                //
                return total;
            }
        }

        /// <summary>Gets whether all components of this <see cref="FuelValue"/> instance are greater than or equal to zero.</summary>
        public bool IsValid
        {
            get
            {
                return (this.Gasoline    >= 0 &&
                        this.Ethanol85   >= 0 &&
                        this.Diesel      >= 0 &&
                        this.Electricity >= 0 &&
                        this.Hydrogen    >= 0 &&
                        this.CNG         >= 0);
            }
        }
        /// <summary>Gets whether all components of this <see cref="FuelValue"/> instance are zero.</summary>
        public bool IsEmpty { get { return this.Equals(FuelValue.Zero); } }
        /// <summary>Gets whether all components of this <see cref="FuelValue"/> instance are <see cref="double.NaN"/>.</summary>
        public bool IsNaN
        {
            get
            {
                return (double.IsNaN(this.Gasoline   ) &&
                        double.IsNaN(this.Ethanol85  ) &&
                        double.IsNaN(this.Diesel     ) &&
                        double.IsNaN(this.Electricity) &&
                        double.IsNaN(this.Hydrogen   ) &&
                        double.IsNaN(this.CNG        ));
            }
        }
        /// <summary>Gets whether all components of this <see cref="FuelValue"/> instance are <see cref="double.PositiveInfinity"/>.</summary>
        public bool IsPositiveInfinity
        {
            get
            {
                return (double.IsPositiveInfinity(this.Gasoline   ) &&
                        double.IsPositiveInfinity(this.Ethanol85  ) &&
                        double.IsPositiveInfinity(this.Diesel     ) &&
                        double.IsPositiveInfinity(this.Electricity) &&
                        double.IsPositiveInfinity(this.Hydrogen   ) &&
                        double.IsPositiveInfinity(this.CNG        ));
            }
        }
        /// <summary>Gets whether all components of this <see cref="FuelValue"/> instance are <see cref="double.NegativeInfinity"/>.</summary>
        public bool IsNegativeInfinity
        {
            get
            {
                return (double.IsNegativeInfinity(this.Gasoline   ) &&
                        double.IsNegativeInfinity(this.Ethanol85  ) &&
                        double.IsNegativeInfinity(this.Diesel     ) &&
                        double.IsNegativeInfinity(this.Electricity) &&
                        double.IsNegativeInfinity(this.Hydrogen   ) &&
                        double.IsNegativeInfinity(this.CNG        ));
            }
        }

        /// <summary>Gets whether this <see cref="FuelValue"/> has positive values for more then one component.</summary>
        public bool IsMultiFuel
        {
            get
            {
                int f = (int)this.FuelType;
                return ((f & (f - 1)) != 0);
            }
        }
        /// <summary>Gets the fuel type of all componenets for this <see cref="FuelValue"/>.</summary>
        public FuelType FuelType
        {
            get
            {
                FuelType fuelType = FuelType.None;
                //
                if (this.Gasoline    > 0) { fuelType |= FuelType.Gasoline   ; }
                if (this.Ethanol85   > 0) { fuelType |= FuelType.Ethanol85  ; }
                if (this.Diesel      > 0) { fuelType |= FuelType.Diesel     ; }
                if (this.Electricity > 0) { fuelType |= FuelType.Electricity; }
                if (this.Hydrogen    > 0) { fuelType |= FuelType.Hydrogen   ; }
                if (this.CNG         > 0) { fuelType |= FuelType.CNG        ; }
                //
                return fuelType;
            }
        }

        /// <summary>Gets the primary component for this <see cref="FuelValue"/>.</summary>
        /// <seealso cref="PrimaryFuel"/>
        public double PrimaryValue { get { return this[this.PrimaryFuel]; } }
        /// <summary>Gets the fuel type of the primary component for this <see cref="FuelValue"/>.
        ///   The primary component is determined in the following order: Gasoline, Diesel, Electricity, Hydrogen, CNG, LNG, LPG,
        ///   Ethanol-85, Biodiesel-20.</summary>
        public FuelType PrimaryFuel
        {
            get
            {
                return // primary fuel types
                       (this.Gasoline    > 0) ? FuelType.Gasoline    :
                       (this.Diesel      > 0) ? FuelType.Diesel      :
                       (this.Electricity > 0) ? FuelType.Electricity :
                       (this.Hydrogen    > 0) ? FuelType.Hydrogen    :
                       (this.CNG         > 0) ? FuelType.CNG         :
                       // secondary fuel types
                       (this.Ethanol85   > 0) ? FuelType.Ethanol85   : FuelType.None;
            }
        }

        #endregion

        #region /*** Variables ***/

        /// <summary>Represents the <see cref="FuelValue"/> whose elements are all <see cref="double.NaN"/>.  This value is
        ///   read-only.</summary>
        public static readonly FuelValue NaN;
        /// <summary>Represents the <see cref="FuelValue"/> whose elements are all 0.  This value is read-only.</summary>
        public static readonly FuelValue Zero;
        /// <summary>Represents the <see cref="FuelValue"/> whose elements are all 1.  This value is read-only.</summary>
        public static readonly FuelValue One;

        #endregion

	}
}
