using System;
using Volpe.Cafe.Generic;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents an object that stores compliance modeling data, as boolean values, for each regulatory class.
    /// </summary>
    [Serializable]
    public class RCBoolean : RCValue<bool>, ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="RCBoolean"/> class.
        /// </summary>
        public RCBoolean() : base() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCBoolean"/> class using the specified initial value for all members.
        /// </summary>
        public RCBoolean(bool initialValue) : base(initialValue, initialValue, initialValue) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCBoolean"/> class using the specified values.
        /// </summary>
        public RCBoolean(bool passengerCar, bool lightTruck, bool lightTruck2b3) : base(passengerCar, lightTruck, lightTruck2b3) { }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="RCBoolean"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCBoolean"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a copy of the current <see cref="RCBoolean"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCBoolean"/>.</returns>
        public new RCBoolean Clone()
        {
            RCBoolean value = new RCBoolean();
            this.CopyTo(value);
            return value;
        }

        #endregion

        /// <summary>
        /// Returns the compliance modeling data value for the specified regulatory class. If <see cref="RegulatoryClass.All"/>
        /// is specified, returns: true, if the compliance modeling data of all regulatory classes is set to true; false,
        /// otherwise.
        /// </summary>
        /// <param name="regClass">The regulatory class for which to obtain the value.</param>
        /// <returns>The compliance modeling data value for the specified regulatory class.</returns>
        public bool GetValue(RegulatoryClass regClass)
        {
            return (regClass == RegulatoryClass.All) ? this.IsTrue : this[regClass];
        }

        #endregion

        #region /*** Properties ***/

        //----- static values -----
        /// <summary>Gets the <see cref="RCBoolean"/> value whose elements are all 0.</summary>
        public static RCBoolean False { get { return new RCBoolean(false); } }
        /// <summary>Gets the <see cref="RCBoolean"/> value whose elements are all 1.</summary>
        public static RCBoolean True { get { return new RCBoolean(true); } }

        //----- instance values -----
        /// <summary>Gets whether the compliance modeling data of all regulatory classes is set to false.</summary>
        public bool IsFalse { get { return this.Equals(False); } }
        /// <summary>Gets whether the compliance modeling data of all regulatory classes is set to true.</summary>
        public bool IsTrue { get { return this.Equals(True); } }

        #endregion

    }
}
