using System;

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Specifies the current state of the modeling process.
    /// </summary>
    [Serializable]
    public enum ModelingState : int
    {
        /// <summary>Specifies that the modeling process has not been initialized yet.</summary>
        Unstarted = 0,
        /// <summary>Specifies that the user has requested to start or resume the modeling process.</summary>
        StartRequested = 1,
        /// <summary>Specifies that the modeling process is currently running.</summary>
        Running = 2,
        /// <summary>Specifies that the user has requested to stop or cancel the modeling process.</summary>
        StopRequested = 3,
        /// <summary>Specifies that the modeling process has been stopped, but did not complete successfully.  This typically
        ///   indicates that the user has requested a stop or a modeling error has occured.</summary>
        Stopped = 4,
        /// <summary>Specifies that the modeling process has completed successfully.</summary>
        Completed = 5
    }

    /// <summary>
    /// Specifies the componnent type of a vehicle technology group.
    /// </summary>
    [Serializable]
    public enum ComponentType : int
    {
        /// <summary>Specifies a general component.  This means that all vehicles should be analyzed for technology applicability.</summary>
        General = 0,
        /// <summary>Specifies that the component represents a vehicle engine.  This means that engine specific vehicles should
        ///   be analyzed for technology applicability.</summary>
        Engine = 1,
        /// <summary>Specifies that the component represents a vehicle transmission.  This means that transmission specific
        ///   vehicles should be analyzed for technology applicability.</summary>
        Transmission = 2,
        /// <summary>Specifies that the component represents a vehicle nameplate.  This means that nameplate specific vehicles
        ///   should be analyzed for technology applicability.</summary>
        Nameplate = 3
    }

    /// <summary>
    /// Specifies which of the model inputs are required for compliance modeling.
    /// </summary>
    [Serializable]
    public enum RequiredModelInputs
    {
        /// <summary>Specifies that none of the model inputs are required.</summary>
        None = 0,

        /// <summary>Specifies that the market data input file is requried for compliance modeling.</summary>
        MarketData = 1,
        /// <summary>Specifies that the technologies input file is requried for compliance modeling.</summary>
        Technologies = 2,
        /// <summary>Specifies that the parameters input file is requried for compliance modeling.</summary>
        Parameters = 4,
        /// <summary>Specifies that the scenarios input file is requried for compliance modeling.</summary>
        Scenarios = 8,

        /// <summary>Specifies that all model inputs are required.</summary>
        All = MarketData | Technologies | Parameters | Scenarios
    }
}
