using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Serialization;
using Volpe.Cafe.IO;
using Volpe.Cafe.Utils;

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Provides an overall view of modeling configuration settings.
    /// </summary>
    [Serializable]
    public sealed class ModelingSettings : ICloneable
    {

        #region /*** Constructors ***/

        // Private constructor for internal usage (cloning).
        ModelingSettings(ModelingSettings prototype)
        {
            this._runtimeSettings = new List<ISettings>();
            for (int i = 0; i < prototype._runtimeSettings.Count; i++)
            {
                this._runtimeSettings.Add(prototype._runtimeSettings[i].Clone());
            }
            //
            this._technologies = (prototype._technologies == null) ? null : prototype._technologies.Clone();
            this._parameters   = (prototype._parameters   == null) ? null : prototype._parameters.Clone();
            this._scenarios    = (prototype._scenarios    == null) ? null : prototype.CloneScenarios(prototype._scenarios);
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="ModelingSettings"/> class.
        /// </summary>
        public ModelingSettings()
        {
            this._runtimeSettings = new List<ISettings>();
            this._runtimeSettings.Add(new InputSettings());
            this._runtimeSettings.Add(new OutputSettings());
            this._runtimeSettings.Add(new OperatingModes());
            //
            // load default settings
            for (int i = 0; i < this._runtimeSettings.Count; i++)
            {
                this._runtimeSettings[i].SetDefaultSettings();
            }
        }

        #endregion


        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current ModelingSettings instance.
        /// </summary>
        /// <returns>A new object that is a copy of this ModelingSettings.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current ModelingSettings instance.
        /// </summary>
        /// <returns>A new object that is a copy of this ModelingSettings.</returns>
        public ModelingSettings Clone()
        {
            return new ModelingSettings(this);
        }
        List<Scenario> CloneScenarios(List<Scenario> list)
        {
            // initialize a new list instance
            List<Scenario> newList = new List<Scenario>(list.Count);
            // traverse the current list and clone each element
            for (int i = 0, count = list.Count; i < count; i++)
            {
                newList.Add((Scenario)(list[i].Clone()));
            }
            // return the new list
            return newList;
        }

        #endregion

        /// <summary>
        /// Summarizes this <see cref="ModelingSettings"/> instance to the specified <see cref="LogWriter"/>.
        /// </summary>
        /// <param name="writer">The <see cref="LogWriter"/> where to write the summary logs.</param>
        public void WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            writer.Summary.WriteLine("---------- Modeling Settings ----------");
            for (int i = 0; i < this._runtimeSettings.Count; i++)
            {
                this._runtimeSettings[i].WriteSummaryLog(writer);
            }
            if (this._technologies != null && this._technologies.TechnologyList.Length > 0)
            {
                string techStr = "--- Technologies Analyzed ---";
                for (int i = 0; i < this._technologies.TechnologyList.Length; i++)
                {
                    TechnologyInfo tech = this._technologies.TechnologyList[i];
                    techStr += "\n  " + tech.Index + " - " + tech.Name;
                }
                writer.Summary.WriteLine(techStr + "\n");
            }
            if (this._scenarios != null && this._scenarios.Count > 0)
            {
                string scenStr = "--- Scenarios Analyzed ---";
                for (int i = 0, scenCount = this._scenarios.Count; i < scenCount; i++)
                {
                    Scenario scen = this._scenarios[i];
                    scenStr += "\n  " + scen.Index + ". " + scen.Description + ((scen.IsBaseline) ? "  (Baseline)" : "");
                }
                writer.Summary.WriteLine(scenStr + "\n");
            }
        }

        /// <summary>
        /// Obtains the runtime settings of the specified <see cref="Type"/>.
        /// </summary>
        /// <param name="type">The <see cref="Type"/> of the object to look for in the runtime settings.</param>
        /// <returns>The runtime settings of the specified <see cref="Type"/>, or null, if the specified <see cref="Type"/> is
        ///   not found.</returns>
        public ISettings GetRuntimeSettings(Type type)
        {
            int index = this.FindRuntimeSettings(type);
            return (index == -1) ? null : this._runtimeSettings[index];
        }
        /// <summary>
        /// Determines whether this instance contains the custom runtime settings of the specified <see cref="Type"/>.
        /// </summary>
        /// <param name="type">The <see cref="Type"/> of the object to look for in the runtime settings.</param>
        /// <returns>Whether this instance contains the custom runtime settings of the specified <see cref="Type"/>.</returns>
        public bool HasCustomRuntimeSettings(Type type)
        {
            return (this.FindCustomRuntimeSettings(type) != -1);
        }
        int FindRuntimeSettings(Type type)
        {
            for (int i = 0; i < this._runtimeSettings.Count; i++)
            {
                if (this._runtimeSettings[i].GetType() == type) { return i; }
            }
            return -1;
        }
        int FindCustomRuntimeSettings(Type type)
        {
            for (int i = 5; i < this._runtimeSettings.Count; i++)
            {
                if (this._runtimeSettings[i].GetType() == type) { return i; }
            }
            return -1;
        }
        /// <summary>
        /// Adds custom runtime settings to this instance, optionally removing the previous value of the same type.
        /// </summary>
        /// <param name="type">The <see cref="Type"/> of runtime settings to add.</param>
        /// <param name="overwritePrevious">true, to remove the previous value, if one exists; false, to keep the previous value.
        ///   If the runtime settings value of the specified type does not exist, a new value will be added; if a previous value
        ///   exists, if overwritePrevious is true, the previous value will be removed, if overwritePrevious is false, the
        ///   previous value will be preserved and a new one will not be added.</param>
        public void AddCustomRuntimeSettings(Type type, bool overwritePrevious)
        {
            // first, search for the previous instance
            int index = this.FindRuntimeSettings(type);
            if (index != -1)
            {   // if there is a previous version, and we do not want to overwrite it, return without any changes
                if (!overwritePrevious) { return; }
                // if we want to overwrite the previous value, remove it first
                this._runtimeSettings.RemoveAt(index);
            }

            // next, create a new instance and add it to the list
            System.Reflection.ConstructorInfo ctor = type.GetConstructor(Type.EmptyTypes);
            ISettings settings = (ISettings)ctor.Invoke(null);
            settings.SetDefaultSettings();
            this._runtimeSettings.Add(settings);
        }
        /// <summary>
        /// Removes custom runtime settings for the specifeid <see cref="Type"/> from this instance.
        /// </summary>
        /// <param name="type">The <see cref="Type"/> of runtime settings to add.</param>
        public void RemoveCustomRuntimeSettings(Type type)
        {
            int index = this.FindRuntimeSettings(type);
            if (index != -1) { this._runtimeSettings.RemoveAt(index); }
        }
        /// <summary>
        /// Removes all custom runtime settings from this instance.
        /// </summary>
        public void RemoveCustomRuntimeSettings()
        {
            for (int i = this._runtimeSettings.Count - 1; i > 4; i--)
            {
                this._runtimeSettings.RemoveAt(i);
            }
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets or sets an object describing information relating to vehicle technologies and adjustment factors for
        ///   all technology classes.</summary>
        /// <exception cref="System.ArgumentNullException">If the value provided is null.</exception>
        public Technologies Technologies
        {
            get { return this._technologies; }
            set
            {
                if (value == null) { throw new ArgumentNullException("Technologies", "The specified value cannot be null."); }
                this._technologies = value;
            }
        }
        /// <summary>Gets or sets the compliance and effects parameters for the modeling system.</summary>
        /// <exception cref="System.ArgumentNullException">If the value provided is null.</exception>
        public Parameters Parameters
        {
            get { return this._parameters; }
            set
            {
                if (value == null) { throw new ArgumentNullException("Parameters", "The specified value cannot be null."); }
                this._parameters = value;
            }
        }
        /// <summary>Gets or sets a list of vehicle scenarios.</summary>
        /// <exception cref="System.ArgumentNullException">If the value provided is null.</exception>
        public List<Scenario> Scenarios
        {
            get { return this._scenarios; }
            set
            {
                if (value == null) { throw new ArgumentNullException("Scenarios", "The specified value cannot be null."); }
                this._scenarios = value;
            }
        }


        /// <summary>Gets the input configuration settings for the modeling system.</summary>
        public InputSettings InputSettings { get { return (InputSettings)this._runtimeSettings[0]; } }
        /// <summary>Gets the output configuration settings for the modeling system.</summary>
        public OutputSettings OutputSettings { get { return (OutputSettings)this._runtimeSettings[1]; } }
        /// <summary>Gets the operating modes settings for the modeling system.</summary>
        public OperatingModes OperatingModes { get { return (OperatingModes)this._runtimeSettings[2]; } }

        #endregion


        #region /*** Variables ***/

        List<ISettings> _runtimeSettings;

        Technologies _technologies;
        Parameters _parameters;
        List<Scenario> _scenarios;

        #endregion

    }
}
